﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/ndd/ndd_Types.h>

namespace nn { namespace ndd
{

/**
 * @brief       すれちがい通信ライブラリを初期化します。
 */
void Initialize() NN_NOEXCEPT;

//! @name 動作に関する機能
//! @{

/**
 * @brief       自律的なすれちがい通信を有効にします。
 * @details     デフォルトは有効です。
 *              設定内容は、電源断を経由しても消失しません。
 */
void EnableAutoCommunication() NN_NOEXCEPT;

/**
 * @brief       自律的なすれちがい通信を無効にします。
 * @details     デフォルトは有効です。
 *              設定内容は、電源断を経由しても消失しません。
 *              省電力化要望を満たすための機能です。
 *              無効にした場合、nddが自律的に制御する通信機能が無効になります。
 *              nddのAPI起因で動作する通信機能 （例： @ref StartDeviceScan） はこの設定の影響を受けません。
 */
void DisableAutoCommunication() NN_NOEXCEPT;

/**
 * @brief       自律的なすれちがい通信に対する設定を取得します。
 * @return      自律的な通信機能が有効として設定されているか
 */
bool IsAutoCommunicationEnabled() NN_NOEXCEPT;

/**
 * @brief       すれちがい通信のネットワーク機能が動作中か否かを返します。
 * @return      動作状態です。
 * @details     通信状態を UI 等に表示することを目的とした機能です。
 *              フライトモード、すれちがい通信以外の通信機能、すれちがい通信の設定、から影響を受け、動作状態は変化します。
 *              プラットフォームにおけるネットワーク機能の占有状態を元に結果を返すため、実際の電波送受信のタイミングと正確に一致しているものではありません。
 */
bool IsNetworkActive() NN_NOEXCEPT;

//! @}

//! @name 送信に関する機能
//! @{

/**
 * @brief       送信データ変更時にシグナルするイベントを取得します。
 * @param[out]  pEvent 結果格納用のイベント
 * @details     イベントのクリアモードは EventClearMode_AutoClear です。
 */
void AcquireSendDataUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       送信データを追加します。
 * @param[in]   sendDataDescription 送信データ
 * @details     @ref SendDataCountMax 個のデータを設定できます。 @ref SendDataCountMax 個を超えて追加した場合、最も昔に追加した送信データが上書きされます。
 *              送信データは @ref ClearSendData による明示的なクリアを行わない限りは消失しません。
 */
void AddSendData(const SendDataDescription& data) NN_NOEXCEPT;

/**
 * @brief       送信データを削除します。
 * @details     設定済みの送信データを全て削除します。
 */
void ClearSendData() NN_NOEXCEPT;

/**
 * @brief       設定済みの送信データを取得します。
 * @param[out]  buffer 結果格納用バッファ
 * @param[in]   offset 複数の送信データが存在する場合の、取得始点
 * @param[in]   count 格納できる結果の数
 * @return      結果の数です。
 */
int GetSendData(SendDataDescription buffer[], int offset, int count) NN_NOEXCEPT;

//! @}

//! @name 受信に関する機能
//! @{

/**
 * @brief       受信時にシグナルするイベントを取得します。
 * @param[out]  pEvent 結果格納用のイベント
 * @details     イベントのクリアモードは EventClearMode_AutoClear です。
 */
void AcquireReceiveDataEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       最新の受信データを示すカウンタ値を取得します。
 * @return      最新の受信データのカウンタ値です。
 * @details     受信データが 0 件の場合、 @ref GetNextReceiveDataCounter と同等の結果が返ります。
 *              実際にデータが存在するか否かは、 @ref GetReceiveData の結果で確認できます。
 */
uint32_t GetCurrentReceiveDataCounter() NN_NOEXCEPT;

/**
 * @brief       最古の有効な受信データを示すカウンタ値を取得します。
 * @return      最古の有効な受信データのカウンタ値です。
 * @details     受信データが 0 件の場合、 @ref GetNextReceiveDataCounter と同等の結果が返ります。
 *              そのカウンタ値は次回の受信時に最古となります。
 *              実際にデータが存在するか否かは、 @ref GetReceiveData の結果で確認できます。
 */
uint32_t GetOldestReceiveDataCounter() NN_NOEXCEPT;

/**
 * @brief       最新の次の受信データを示すカウンタ値を取得します。
 * @return      最新の次の受信データのカウンタ値です。
 * @details     次のデータを受信するまで当該のカウンタ値は無効です。
 */
uint32_t GetNextReceiveDataCounter() NN_NOEXCEPT;

/**
 * @brief       新しい受信データを指定件数取得可能なカウンタ値を取得します。
 * @param[in]   counter カウンタ値
 * @param[in]   count   件数
 * @return      指定件数の受信データを取得可能なカウンタ値です。
 * @details     新しい受信データを、最大 count 件取得可能なカウンタ値を取得します。
 *              指定したカウンタ値よりも古い受信データを指すカウンタ値は含まれません。
 *              無効な始点となるカウンタ値を指定した場合と、 count に 0 を指定した場合は @ref GetNextReceiveDataCounter と同等の結果が返ります。
 */
uint32_t GetRecentReceiveDataCounter(uint32_t counter, int count) NN_NOEXCEPT;

/**
 * @brief       取得可能な受信データの数を取得します。
 * @param[in]   counter カウンタ値
 * @return      取得可能な受信データの数です。
 * @details     カウンタ値を指定して、取得可能な受信データの数を取得します。
 *              指定したカウンタ値の受信データも数に含まれます。
 */
int GetAvailableReceiveDataCount(uint32_t counter) NN_NOEXCEPT;

/**
 * @brief       受信データを複数件取得します。
 * @param[out]  buffer 受信データ用のバッファ
 * @param[out]  pNextCounter 次回取得に用いるカウンタ値
 * @param[in]   counter カウンタ値
 * @param[in]   count   格納できる結果の数
 * @return      結果の数
 * @details     指定したカウンタ値の受信データ、およびそこから後続する受信データを取得します。
 *              カウンタを演算した場合は、正しい値を取得できない可能性があります。
 *              指定したカウンタ値のデータが既に消えている場合、有効なデータの中で最も古いデータから順に取得されます。
 *              存在しないカウンタ値を指定した場合、結果の数が 0 となります。その際、 pNextCounter には @ref GetNextReceiveDataCounter と同等の結果が出力されます。
 */
int GetReceiveData(ReceiveDataDescription buffer[], uint32_t* pNextCounter, uint32_t counter, int count) NN_NOEXCEPT;

/**
 * @brief       受信データを追加します。
 * @param[in]   data 受信データ
 * @details     最新の受信データとして、指定のデータを追加します。
 *              通常の受信と同様に、受信に関する各機能は動作します。イベントはシグナルされ、最新のカウンタ値が更新されます。
 *              DataId によるフィルタは行われません。
 */
void AddReceiveData(const ReceiveDataDescription& data) NN_NOEXCEPT;

/**
 * @brief       受信データを全件削除します。
 */
void ClearReceiveData() NN_NOEXCEPT;

/**
 * @brief       重複データ判別用のフィルタリストを全件削除します。
 * @details     ndd は、受信データの @ref DataId を保存することで、同じ @ref DataId を持つデータの受信をフィルタしています。
 *              本 API では、このフィルタリストを全件削除し、再度の受信を可能とします。
 */
void ClearDataIdFilter() NN_NOEXCEPT;

//! @}

//! @name デバイススキャンに関する機能
//! @{

/**
 * @brief       すれちがいスキャン完了時にシグナルするイベントを取得します。
 * @param[out]  pEvent 結果格納用のイベント
 * @details     イベントのクリアモードは EventClearMode_AutoClear です。
 */
void AcquireDeviceScanEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       デバイススキャン向けに、メモリを提供します。
 * @param[in]   handle TransferMemoryのハンドル
 * @details     @ref StartDeviceScan の動作中は、メモリが提供されている必要があります。
 *              スキャンで取得したい結果1件につき、約 2KB の提供が必要です。
 *              メモリが足りない場合、スキャン結果が少なくなります。
 *              多重に提供できません。
 */
void LendDeviceScanMemory(nn::os::NativeHandle handle) NN_NOEXCEPT;

/**
 * @brief       デバイススキャン向けの、提供済みメモリの利用を停止させます。
 * @param[in]   handle TransferMemoryのハンドル
 * @details     未提供のメモリに対して停止はできません。
 *              スキャン動作中は、メモリ利用の停止はできません。
 *              この場合、先に @ref CancelDeviceScan を呼んでください。
 */
void PullbackDeviceScanMemory(nn::os::NativeHandle handle) NN_NOEXCEPT;

/**
 * @brief       すれちがい通信動作中の周囲のデバイスをスキャンします。
 * @details     数秒間スキャンし、自動で終了します。
 *              APIはスキャンとは同期せずに、すぐ返ります。
 *              スキャン中に何らかの要因でスキャンが停止した場合、その時点までの結果を取得できます。
 *              スキャン完了前に再度呼び出した場合、新しいスキャンは開始されません。
 *              事前に @ref LendDeviceScanMemory で作業用のメモリを提供する必要があります。
 *              [暫定仕様]メモリの提供無しで動作します。
 */
void StartDeviceScan() NN_NOEXCEPT;

/**
 * @brief       動作中のデバイススキャンを停止させます。
 * @details     デバイススキャンの停止タイミングは、イベントのシグナルで判断できます。
 *              キャンセルした場合も、スキャン完了と同じようにイベントがシグナルし、それまでの結果を取得できます。
 *              @ref StartDeviceScan の処理が返る前にこのAPIを呼んだ場合、キャンセルされません。
 */
void CancelDeviceScan() NN_NOEXCEPT;

/**
 * @brief       デバイススキャンの結果を取得します。
 * @param[out]  buffer 結果格納用バッファ
 * @param[in]   count 格納できる結果の数
 * @return      結果の数
 * @details     当該イベントのシグナル前に呼び出した場合、前回の結果が返ります。
 *              一回のスキャン内での重複情報はフィルタされます。重複の基準は、 ReceiveData.data の完全一致です。
 *              ソートは行われません。
 */
int GetDeviceScanResult(ReceiveDataDescription buffer[], int count) NN_NOEXCEPT;

//! @}

}} // namespace nn::ndd
