﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/account/account_Types.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/nd/nd_TypesForSystem.h>
#include <nn/nd/detail/nd_SystemSaveData.h>
#include <nn/nd/service/nd_UserIdManager.h>
#include <nn/ndd/ndd_Types.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/settings/settings_Language.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace nd { namespace service {

class SendDataManager
{
private:
    class UserData
    {
    public:
        account::Uid m_Uid = account::InvalidUid;
        account::Nickname m_Nickname;
        int m_RecentlyPlayedApplicationCount;
        ncm::ApplicationId m_RecentlyPlayedApplication[RecentlyPlayedApplicationCountMax];

        ndd::DataId m_DataId;
        size_t m_SystemDataSize;
        Bit8 m_SystemData[SystemDataSizeMax];

        UserData() NN_NOEXCEPT {};
        void Reset() NN_NOEXCEPT;
        void Initialize(const account::Uid& uid) NN_NOEXCEPT;
        bool IsInitialized() const NN_NOEXCEPT;

        void SetDataId(const ndd::DataId& dataId) NN_NOEXCEPT;

        //! 更新された場合に true を返す。
        bool RefreshAccountProfile() NN_NOEXCEPT;
        bool RefreshRecentlyPlayedApplication() NN_NOEXCEPT;

        void SetSystemData(const void* data, size_t dataSize) NN_NOEXCEPT;
        void ClearSystemData() NN_NOEXCEPT;

        //! 未実装
        Result Store() NN_NOEXCEPT;
    private:
        //! 未実装
        Result Load() NN_NOEXCEPT;
    };

public:
    explicit SendDataManager(UserIdManager& userIdManager) NN_NOEXCEPT;
    void Initialize(const util::Span<const account::Uid>& uids) NN_NOEXCEPT;
    void Update(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT;

    void CheckNetworkUserIdUpdate() NN_NOEXCEPT;

    void RefreshAccountProfile() NN_NOEXCEPT;
    void RefreshRecentlyPlayedApplication() NN_NOEXCEPT;
    void RefreshSendDataId() NN_NOEXCEPT;

    void SetSender(const account::Uid& uid) NN_NOEXCEPT;
    bool GetSender(account::Uid* pOut) const NN_NOEXCEPT;

    void SetDataId(const account::Uid& uid, const ndd::DataId& dataId) NN_NOEXCEPT;
    void SetSystemData(const account::Uid& uid, const void* data, size_t size) NN_NOEXCEPT;
    void ClearSystemData(const account::Uid& uid) NN_NOEXCEPT;

    os::SystemEvent& GetSendDataIdUpdateScheduleEvent() NN_NOEXCEPT;

    void GetSendDataForDebug(ndd::SendDataDescription *pOut, const account::Uid& uid) NN_NOEXCEPT;

private:
    void UpdateSendDataUnsafe(const UserData& userData) NN_NOEXCEPT;
    void UpdateUnsafe(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT;
    UserData* FindOrInitUserDataUnsafe(const account::Uid& uid) NN_NOEXCEPT;

    Result StoreSettings() NN_NOEXCEPT;
    Result LoadSettings() NN_NOEXCEPT;

    mutable os::SdkMutex m_Mutex;
    account::Uid m_Sender = account::InvalidUid;
    util::optional<NetworkUserId> m_SenderNetworkUserIdCache;
    settings::LanguageCode m_SystemLanguageCode;
    std::array<UserData, account::UserCountMax> m_UserDatas;

    time::SteadyClockTimePoint m_SendDataIdRefreshTimePoint;
    bgtc::Task m_SendDataIdUpdateTask;

    const char* SettingsFilePath = "settings.bin";
    detail::SystemSaveData m_SystemSaveData;
    struct SettingsDataLayout
    {
        account::Uid sender;
        time::SteadyClockTimePoint sendDataIdRefreshTimePont;
    };
    NN_STATIC_ASSERT(sizeof(SettingsDataLayout) == 40);

    UserIdManager& m_UserIdManager;
};

}}} // ~namespace nn::nd::service
