﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief 本体機能向けの機能
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/account/account_Types.h>
#include <nn/nd/nd_NeighborInfoForSystem.h>
#include <nn/nd/nd_TypesForSystem.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace nd { namespace service {
class IScannerForSystem;
}}}

namespace nn { namespace nd {

/**
    @brief      ライブラリを本体機能向けの権限で初期化します。
*/
void InitializeForSystem() NN_NOEXCEPT;

//! @name 送信データの設定に関する機能（本体機能向け）
//! @{

/**
    @brief      指定したユーザーの送信データに含める本体機能用データを設定します。
    @param[in]  uid ユーザー
    @param[in]  data データ
    @param[in]  dataSize データサイズ
    @pre
                - uid が指すユーザーが本体上に存在する
                - data != nullptr
                - 0 < dataSize <= @ref nd::SystemDataSizeMax
*/
void SetSystemData(const account::Uid& uid, const void* data, size_t dataSize) NN_NOEXCEPT;

/**
    @brief      @ref SetSystemData で設定した本体機能用データを削除します。
    @param[in]  uid ユーザー
    @details    データが設定されていない場合は何もしません。
    @pre
                - uid が指すユーザーが本体上に存在する
*/
void ClearSystemData(const account::Uid& uid) NN_NOEXCEPT;

/**
    @brief      データを送信するユーザーを設定します。
    @param[in]  uid ユーザー
    @pre
                - uid が指すユーザーが本体上に存在する
*/
void SetSender(const account::Uid& uid) NN_NOEXCEPT;

/**
    @brief      データを送信するユーザーを取得します。
    @param[out] pOut ユーザー
    @return     取得の成否
    @pre
                - pOut != nullptr
    @details    一度も @ref SetSender が呼び出されていない場合や、@ref SetSender で設定したユーザーが削除された場合など、
                データを送信するユーザーが存在しない場合に返り値が false になります。返り値が false の場合の *pOut の値は不定です。
*/
bool GetSender(account::Uid* pOut) NN_NOEXCEPT;

//! @}


//! @name 受信データの取得に関する機能（本体機能向け）
//! @{

/**
    @brief      新しいデータを受信したときにシグナルされるシステムイベントを取得します。
    @return     新しいデータを受信したときにシグナルされるシステムイベントのポインタ
    @details    新しいデータを受信したときにシグナルされるシステムイベントを取得します。
                このシステムイベントのクリアモードは nn::os::EventClearMode_Manual に設定されます。
                シグナルは自動的にはクリアされません。新しいデータの受信を待ち受ける状態に入る前に明示的にクリアしてください。
*/
os::SystemEvent* GetNeighborInfoUpdateEventForSystem() NN_NOEXCEPT;

/**
    @brief      指定したユーザーが利用可能な受信データの数を取得します。
    @param[in]  uid ユーザー
    @return     データ数
    @details    新しいデータの受信や @ref ReceiveNeighborInfoForSystem の呼び出しによって取得される値は変化します。
    @pre
                - uid が指すユーザーが本体上に存在する
*/
int GetReceivableNeighborInfoCountForSystem(const account::Uid& uid) NN_NOEXCEPT;

/**
    @brief      ユーザーを指定して受信データを取得します。
    @param[in]  out 受信データの格納先
    @param[in]  outCount out の要素数
    @param[in]  uid ユーザー
    @return     取得した情報の数
    @pre
                - out != nullptr
                - outCount > 0
                - uid が指すユーザーが本体上に存在する
    @details    受信データがどこまで取得されたかはシステムによってユーザー毎に管理されます。@n
                同じユーザーを指定して本関数を呼び出した場合、前回の呼び出しで取得したデータの続きからデータを取得します。一度取得したデータの再取得はできません。
*/
int ReceiveNeighborInfoForSystem(NeighborInfoForSystem out[], int outCount, const account::Uid& uid) NN_NOEXCEPT;

//! @}

//! @name 周囲のデータのスキャンに関する機能（本体機能向け）
//! @{

/**
    @brief      周囲のデータのスキャンを行うためのクラス。
    @details    スキャンでは検出可能な全てのデータを、そのデータが過去一定時間以内に検出済みかどうかに関わらず取得します。
                そのため、スキャンを連続して実行した場合には同じデータが結果に含まれることがあります。
*/
class ScannerForSystem
{
    NN_DISALLOW_COPY(ScannerForSystem);
    friend ScannerForSystem CreateScannerForSystem(void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

public:
    /**
        @brief      デフォルトコンストラクタ。
        @details    デフォルトコンストラクタで作成したオブジェクトは無効な状態です。
                    API の呼び出しが可能な有効なオブジェクトを作成するには @ref CreateScannerForSystem を使用してください。
    */
    ScannerForSystem() NN_NOEXCEPT = default;

    /**
        @brief      ムーブコンストラクタ。
        @param[in]  rhs 移動元のオブジェクト
    */
    ScannerForSystem(ScannerForSystem&& rhs) NN_NOEXCEPT;

    /**
        @brief      オブジェクトを破棄して無効な状態にします。
    */
    ~ScannerForSystem() NN_NOEXCEPT;

    /**
        @brief      ムーブ代入演算子。
        @param[in]  rhs 移動元のオブジェクト
        @return     このオブジェクトへの参照
    */
    ScannerForSystem& operator=(ScannerForSystem&& rhs) NN_NOEXCEPT;

    /**
        @brief      オブジェクトを入れ替えます。
        @param[in]  rhs 入れ替え対象のオブジェクト
    */
    void swap(ScannerForSystem& rhs) NN_NOEXCEPT;

    /**
        @brief      オブジェクトが有効かどうかを取得します。
        @return     オブジェクトが有効かどうか
    */
    bool IsValid() const NN_NOEXCEPT;

    /**
        @brief      周囲のデータのスキャンを開始します。
        @details    既にスキャンを開始していた場合、新しいスキャンは開始されません。
    */
    void StartScan() NN_NOEXCEPT;

    /**
        @brief      周囲のデータのスキャンをキャンセルします。
        @details    スキャンをキャンセルした場合にも GetSystemEvent() で取得できるイベントはシグナルされ、
                    スキャンを開始してからキャンセルされるまでの結果が GetScanResult で取得できます。
                    スキャンが開始されていなかった場合や既にスキャンが完了済みであった場合にはキャンセル処理は行われず、イベントもシグナルされません。
    */
    void CancelScan() NN_NOEXCEPT;

    /**
        @brief      周囲のデータのスキャンの結果を取得します。
        @param[out] out スキャン結果の格納先
        @param[in]  outCount out の要素数
        @param[in]  uid ユーザー
        @return     取得したスキャン結果の個数
        @pre
                    - スキャンが完了済み
        @details    （仮仕様）現在の受け取れる結果の最大数は固定値で 50 です。
                    将来的には @ref CreateScannerForSystem で指定した workBufferSize に依存した個数になります。
    */
    int GetScanResult(NeighborInfoForSystem out[], int outCount, const account::Uid& uid) NN_NOEXCEPT;

    /**
        @brief      周囲のデータのスキャンが完了した際にシグナルされるシステムイベントのポインタを取得します。
        @return     周囲のデータのスキャンが完了した際にシグナルされるシステムイベントのポインタ
        @details    周囲のデータのスキャンが完了した際にシグナルされるシステムイベントのポインタを取得します。
                    このシステムイベントのクリアモードは nn::os::EventClearMode_Manual に設定されます。
                    また、StartScan() を呼び出した際には自動的にクリアされます。
    */
    os::SystemEvent* GetSystemEvent() NN_NOEXCEPT;

private:
    explicit ScannerForSystem(service::IScannerForSystem* ptr) NN_NOEXCEPT;
    service::IScannerForSystem* m_Ptr = nullptr;
    util::optional<os::SystemEvent> m_pSystemEvent;
    bool m_IsSystemEventHandleManaged = false;
};

void swap(ScannerForSystem& o0, ScannerForSystem& o1) NN_NOEXCEPT;

/**
    @brief      周囲のデータをスキャンするためのオブジェクトを作成します。
    @param[in]  workBuffer 周囲のデータの取得に使用するバッファ
    @param[in]  workBufferSize workBuffer のサイズ
    @return     周囲のデータをスキャンするためのオブジェクト
    @pre
                - 有効な状態の ScannerForSystem が他に存在していない
                - workBuffer != nullptr
                - workBuffer のアドレスが @ref nn::os::MemoryPageSize の整数倍
                - workBufferSize が @ref nn::os::MemoryPageSize の整数倍
    @details    （仮仕様）現在は workBuffer, workBufferSize は事前条件のチェックのみが行われます。実際には使用されません。
                将来的にはここで指定したバッファによってスキャンの結果で取得できるデータの個数が決まるようになります。
*/
ScannerForSystem CreateScannerForSystem(void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
    @brief      （未実装）指定した個数のデータをスキャンで取得可能にするために必要なバッファーのサイズを取得します。
    @param[in]  count データの個数
    @return     count 個のデータを取得できる ScannerForSystem を作成するために @ref CreateScannerForSystem に渡す必要があるバッファのサイズ。
    @pre
                - count > 0
    @details    （未実装）現在は常に 0 が返ります。
*/
size_t GetRequiredBufferSizeForScannerForSystem(int count) NN_NOEXCEPT;

//! @}

}} // ~namespace nn::nd
