﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_Types.h>
#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/migration_UserMigrationServiceTypes.h>
#include <nn/migration/detail/migration_AsyncExecutionResource.h>
#include <nn/migration/detail/migration_LoginSession.h>
#include <nn/migration/detail/migration_ServiceObjectAllocator.h>
#include <nn/migration/detail/migration_ThreadResourceAllocator.h>
#include <nn/migration/user/migration_UserMigrationInternalTypes.h>
#include <nn/migration/user/migration_UserMigrationStateManager.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace migration { namespace user {
class IServer;
class IClient;
}}} // ~namespace nn::migration::user

namespace nn { namespace migration { namespace user {

template <typename ImplConfiguration>
class ServiceImpl
    : public sf::IServiceObject
{
private:
    // 実装の定義
    typedef typename ImplConfiguration::StateStoragePolicy StateStoragePolicy;
    typedef typename ImplConfiguration::ServerImpl ServerImpl;
    typedef typename ImplConfiguration::ClientImpl ClientImpl;

    // Server, Client, 全ての AsyncContext を生成するためのメモリサイズ
    static const size_t AllocatorHeapSize = 64 * 1024;
    typedef detail::ServiceObjectAllocator<AllocatorHeapSize> ObjectAllocator;

    // 静的リソース
    ObjectAllocator m_ObjectAllocator;
    detail::ThreadResourceAllocator& m_ThreadResourceAllocator;
    typename StateStoragePolicy::Storage m_StateStorage;
    StateManager m_StateManager;
    detail::ClientLoginSessionHolder m_ClientLoginSessionHolder;

public:
    explicit ServiceImpl(detail::ThreadResourceAllocator& allocator, void* loginSessionBuffer, size_t loginSessionBufferSize) NN_NOEXCEPT;

    Result TryGetLastMigrationInfo(sf::Out<bool> pOutResult, sf::Out<LastMigrationInfo> pOutData) NN_NOEXCEPT;

    Result CreateServer(
        sf::Out<sf::SharedPointer<IServer>> pOut,
        const account::Uid& uid, const UserMigrationServerProfile& profile, sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT;
    Result ResumeServer(
        sf::Out<sf::SharedPointer<IServer>> pOut,
        sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT;

    Result CreateClient(
        sf::Out<sf::SharedPointer<IClient>> pOut,
        const UserMigrationClientProfile& profile, sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT;
    Result ResumeClient(
        sf::Out<sf::SharedPointer<IClient>> pOut,
        sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::user

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/user/migration_UserMigrationClientProxy.h>
#include <nn/migration/user/migration_UserMigrationServerProxy.h>

namespace nn { namespace migration { namespace user {

template <typename ImplConfiguration>
inline ServiceImpl<ImplConfiguration>::ServiceImpl(detail::ThreadResourceAllocator& allocator, void* loginSessionBuffer, size_t loginSessionBufferSize) NN_NOEXCEPT
    : m_ThreadResourceAllocator(allocator)
    , m_StateManager(m_StateStorage)
    , m_ClientLoginSessionHolder(loginSessionBuffer, loginSessionBufferSize)
{
}

template <typename ImplConfiguration>
inline Result ServiceImpl<ImplConfiguration>::TryGetLastMigrationInfo(sf::Out<bool> pOutResult, sf::Out<LastMigrationInfo> pOutData) NN_NOEXCEPT
{
    return m_StateManager.TryGetLastMigrationInfo(pOutResult.GetPointer(), pOutData.GetPointer());
}

template <typename ImplConfiguration>
inline Result ServiceImpl<ImplConfiguration>::CreateServer(
    sf::Out<sf::SharedPointer<IServer>> pOut,
    const account::Uid& uid, const UserMigrationServerProfile& profile, sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT
{
    // 引数の検査
    NN_MIGRATION_DETAIL_THROW_UNLESS(uid, detail::ResultInvalidUid());
    typedef ServerProxy<ObjectAllocator, ServerImpl> ServerProxyType;
    NN_MIGRATION_DETAIL_THROW_UNLESS(workMemorySize >= ServerProxyType::RequiredWorkMemorySize, detail::ResultInsufficientMemory());

    // 状態の作成
    StateController* pController;
    NN_MIGRATION_DETAIL_DO(m_StateManager.StartServer(&pController));

    // サーバーオブジェクトの生成
    auto p = ObjectAllocator::Factory::template CreateSharedEmplaced<IServer, ServerProxyType>(
        &m_ObjectAllocator,
        typename ServerProxyType::ParentPtr(this, true), m_ObjectAllocator.GetShared(), m_ThreadResourceAllocator.Allocate(),
        std::move(workMemory), workMemorySize,
        *pController, m_ThreadResourceAllocator.Allocate());
    NN_MIGRATION_DETAIL_THROW_UNLESS(p, detail::ResultOutOfSessionObject());

    // サーバーオブジェクトの初期化
    NN_MIGRATION_DETAIL_DO(p.GetImpl().Initialize(uid, profile));

    // OK
    *pOut = p;
    NN_RESULT_SUCCESS;
}

template <typename ImplConfiguration>
inline Result ServiceImpl<ImplConfiguration>::ResumeServer(
    sf::Out<sf::SharedPointer<IServer>> pOut,
    sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT
{
    // 引数の検査
    typedef ServerProxy<ObjectAllocator, ServerImpl> ServerProxyType;
    NN_MIGRATION_DETAIL_THROW_UNLESS(workMemorySize >= ServerProxyType::RequiredWorkMemorySize, detail::ResultInsufficientMemory());

    // 状態の復元
    StateController* pController;
    NN_MIGRATION_DETAIL_DO(m_StateManager.ResumeServer(&pController));

    // サーバーオブジェクトの生成
    auto p = ObjectAllocator::Factory::template CreateSharedEmplaced<IServer, ServerProxyType>(
        &m_ObjectAllocator,
        typename ServerProxyType::ParentPtr(this, true), m_ObjectAllocator.GetShared(), m_ThreadResourceAllocator.Allocate(),
        std::move(workMemory), workMemorySize,
        *pController, m_ThreadResourceAllocator.Allocate());
    NN_MIGRATION_DETAIL_THROW_UNLESS(p, detail::ResultOutOfSessionObject());

    // サーバーオブジェクトの再開
    NN_MIGRATION_DETAIL_DO(p.GetImpl().Resume());

    // OK
    *pOut = p;
    NN_RESULT_SUCCESS;
}

template <typename ImplConfiguration>
inline Result ServiceImpl<ImplConfiguration>::CreateClient(
    sf::Out<sf::SharedPointer<IClient>> pOut,
    const UserMigrationClientProfile& profile, sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT
{
    // 引数の検査
    typedef ClientProxy<ObjectAllocator, ClientImpl> ClientProxyType;
    NN_MIGRATION_DETAIL_THROW_UNLESS(workMemorySize >= ClientProxyType::RequiredWorkMemorySize, detail::ResultInsufficientMemory());

    // 状態の作成
    StateController* pController;
    NN_MIGRATION_DETAIL_DO(m_StateManager.StartClient(&pController));

    // クライアントオブジェクトの生成
    auto p = ObjectAllocator::Factory::template CreateSharedEmplaced<IClient, ClientProxyType>(
        &m_ObjectAllocator,
        typename ClientProxyType::ParentPtr(this, true), m_ObjectAllocator.GetShared(), m_ThreadResourceAllocator.Allocate(),
        std::move(workMemory), workMemorySize,
        *pController, m_ClientLoginSessionHolder.Acquire(), m_ThreadResourceAllocator.Allocate());
    NN_MIGRATION_DETAIL_THROW_UNLESS(p, detail::ResultOutOfSessionObject());

    // クライアントオブジェクトの初期化
    NN_MIGRATION_DETAIL_DO(p.GetImpl().Initialize(profile));

    // OK
    *pOut = p;
    NN_RESULT_SUCCESS;
}

template <typename ImplConfiguration>
inline Result ServiceImpl<ImplConfiguration>::ResumeClient(
    sf::Out<sf::SharedPointer<IClient>> pOut,
    sf::NativeHandle&& workMemory, uint32_t workMemorySize) NN_NOEXCEPT
{
    // 引数の検査
    typedef ClientProxy<ObjectAllocator, ClientImpl> ClientProxyType;
    NN_MIGRATION_DETAIL_THROW_UNLESS(workMemorySize >= ClientProxyType::RequiredWorkMemorySize, detail::ResultInsufficientMemory());

    // 状態の復元
    StateController* pController;
    NN_MIGRATION_DETAIL_DO(m_StateManager.ResumeClient(&pController));

    // クライアントオブジェクトの生成
    auto p = ObjectAllocator::Factory::template CreateSharedEmplaced<IClient, ClientProxyType>(
        &m_ObjectAllocator,
        typename ClientProxyType::ParentPtr(this, true), m_ObjectAllocator.GetShared(), m_ThreadResourceAllocator.Allocate(),
        std::move(workMemory), workMemorySize,
        *pController, m_ClientLoginSessionHolder.Acquire(), m_ThreadResourceAllocator.Allocate());
    NN_MIGRATION_DETAIL_THROW_UNLESS(p, detail::ResultOutOfSessionObject());

    // クライアントオブジェクトの再開
    NN_MIGRATION_DETAIL_DO(p.GetImpl().Resume());

    // OK
    *pOut = p;
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::user
