﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_IdcClient.h>
#include <nn/migration/detail/migration_SaveDataTransfer.h>
#include <nn/migration/user/migration_UserMigrationContext.h>
#include <nn/migration/user/migration_UserMigrationProgressMonitor.h>
#include <nn/migration/user/migration_UserMigrationStateController.h>

namespace nn { namespace migration { namespace user {

template <typename ConnectionType, typename EncryptionPolicy>
class IdcClient
    : public detail::IdcClient<ConnectionType, EncryptionPolicy>
{
private:
    typedef detail::IdcClient<ConnectionType, EncryptionPolicy> Base;

    struct Resource
    {
        detail::SaveDataImporter::ResourceStorage dataTransferWorkBuffer;
    } m_Resource;

public:
    explicit IdcClient(detail::ThreadResource&& threadResource) NN_NOEXCEPT
        : Base(std::move(threadResource))
    {
    }
    Result ImportUserMigrationServerInfo(ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result MatchUserMigrationServerInfo(const ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result ExportUserMigrationClientInfo(const ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result WaitUserMigrationClientAccepted(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result SynchronizeStateInInitialization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result SynchronizeStateInFinalization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result SynchronizeStateFinalized(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result Suspend(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result ImportMigrationList(ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result ImportSaveData(detail::SaveDataImporter& importer, int index, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::user


#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Settings.h>
#include <nn/migration/user/migration_UserMigrationIdcImpl.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace user {

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::ImportUserMigrationServerInfo(ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Importing user migration server info\n");
    GetUserMigrationServerInfoClientApi api;
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    return api.ImportServerInfo(context);
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::MatchUserMigrationServerInfo(const ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Matching user migration server info\n");
    GetUserMigrationServerInfoClientApi api;
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    return api.MatchServerInfo(context);
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::ExportUserMigrationClientInfo(const ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Exporting user migration client info\n");
    PutUserMigrationClientInfoClientApi api(context);
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::WaitUserMigrationClientAccepted(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Waiting for connection accepted by server\n");
    WaitUserMigrationStartClientApi api;
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetUiAcceptanceWaitTimeoutSeconds(), pCancellable));
    return api.GetResult();
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::SynchronizeStateInInitialization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Synchronizing state to InInitialization\n");
    NN_SDK_ASSERT(!stateController.IsResumable());
    SynchronizeStateClientApi api(MigrationState_InInitialization);
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_DO(stateController.SetStateInInitialization());
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::SynchronizeStateInFinalization(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Synchronizing state to InFinalization\n");
    if (!stateController.IsTransferDone())
    {
        NN_RESULT_DO(stateController.SetStateInFinalization());
    }
    SynchronizeStateClientApi api(MigrationState_InFinalization);
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::SynchronizeStateFinalized(StateController& stateController, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Synchronizing state to Finalized\n");
    NN_RESULT_DO(stateController.SetStateFinalized());
    SynchronizeStateClientApi api(MigrationState_Finalized);
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::Suspend(const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Suspending user migration\n");
    NotifySuspendClientApi api;
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::ImportMigrationList(ClientContext& context, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[Client] Importing migration list\n");
    GetMigrationListClientApi api(context);
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferSmallDataTimeoutSeconds(), pCancellable));
    return api.Adapt();
}

template <typename ConnectionType, typename EncryptionPolicy>
Result IdcClient<ConnectionType, EncryptionPolicy>::ImportSaveData(detail::SaveDataImporter& importer, int index, const ConnectionType& connection, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    importer.Initialize(*this, m_Resource.dataTransferWorkBuffer);
    RequestTransferClientApi<detail::SaveDataImporter> api(importer, index);
    NN_RESULT_DO(Base::InvokeUserCommand(connection, api, detail::GetTransferLargeDataTimeoutSeconds(), pCancellable));
    NN_RESULT_DO(importer.Complete());
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::user
