﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/detail/migration_AbstractStorage.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_MigrationList.h>
#include <nn/migration/detail/migration_MigrationProgress.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace user {

struct ServerInfo
{
    uint32_t fwVersion;
    util::Uuid sessionId;
    account::Uid user;
    UserMigrationServerProfile profile;
    struct
    {
        UserMigrationClientProfile profile;
    } clientInfo;
};
struct ServerInfoForTransfer // TODO Endian
{
    uint32_t fwVersion;
    util::Uuid sessionId;
    account::Uid user;
    UserMigrationServerProfile profile;
};

struct ClientInfo
{
    account::NetworkServiceAccountId nsaId;
    UserMigrationClientProfile profile;
    struct
    {
        uint32_t fwVersion;
        util::Uuid sessionId;
        account::Uid user;
        UserMigrationServerProfile profile;
    } serverInfo;
};
struct ClientInfoForTransfer // TODO Endian
{
    UserMigrationClientProfile profile;
};

class ServerContext
{
private:
    const detail::AbstractStorage& m_Storage;
    util::optional<detail::MigrationListHeader> m_pListHeader;
    util::optional<ServerInfo> m_pInfo;

public:
    explicit ServerContext(const detail::AbstractStorage& storage) NN_NOEXCEPT
        : m_Storage(storage)
    {
    }

    // 初期化
    void CreateMigrationInfo(const account::Uid& uid, const UserMigrationServerProfile& serverProfile) NN_NOEXCEPT;
    bool TryResume() NN_NOEXCEPT;

    // ユーザー移行情報の取得
    account::Uid GetUid() const NN_NOEXCEPT;
    void GetServerProfile(UserMigrationServerProfile* pOut) const NN_NOEXCEPT;
    void GetServerInfoForTransfer(ServerInfoForTransfer* pOut) const NN_NOEXCEPT;

    Result ImportClientInfo(const ClientInfoForTransfer& clientInfo) NN_NOEXCEPT;
    bool MatchClientInfo(const ClientInfoForTransfer& clientInfo) const NN_NOEXCEPT;
    void GetClientProfile(UserMigrationClientProfile* pOut) const NN_NOEXCEPT;

    // 移行リストの作成
    Result CreateMigrationList(void* buffer, size_t bufferSize, const detail::Cancellable* pCancellable) NN_NOEXCEPT;
    size_t GetMigrationListSize() const NN_NOEXCEPT;
    size_t ReadMigrationList(detail::DataInfo* pOut, size_t count, size_t offset) const NN_NOEXCEPT;
    size_t ReadMigrationListForExport(void* buffer, size_t bufferSize, size_t offset) const NN_NOEXCEPT;
};

class ClientContext
{
private:
    const detail::AbstractStorage& m_Storage;
    ncm::ApplicationId* m_RelatedApps;
    size_t m_RelatedAppCapacity;
    size_t m_RelatedAppCount;

    util::optional<detail::MigrationListHeader> m_pListHeader;
    util::optional<detail::SimpleMigrationProgress> m_pProgress;
    util::optional<ClientInfo> m_pInfo;

    size_t m_TemporaryMigrationListSize;
    size_t m_TemporaryMigrationListOffset;

    TransferInfo CalculateTotalTransferInfo(void* buffer, size_t bufferSize) const NN_NOEXCEPT;
    Result ResetProgressInfo(const detail::MigrationListHeader& listHeader, void* buffer, size_t bufferSize) NN_NOEXCEPT;

public:
    explicit ClientContext(const detail::AbstractStorage& storage) NN_NOEXCEPT
        : m_Storage(storage)
        , m_RelatedApps(nullptr)
    {
    }

    // 初期化
    void Initialize(void* transferredAppBuffer, size_t transferredAppBufferSize) NN_NOEXCEPT;

    // 新規作成
    void CreateMigrationInfo(const UserMigrationClientProfile& clientProfile) NN_NOEXCEPT;
    void SetNetworkServiceAccountId(const account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT;
    Result ImportServerInfo(const ServerInfoForTransfer& serverInfo) NN_NOEXCEPT;

    // 再開
    bool TryResume() NN_NOEXCEPT;
    bool MatchServerInfo(const ServerInfoForTransfer& serverInfo) const NN_NOEXCEPT;

    // ユーザー移行情報の取得
    account::NetworkServiceAccountId GetNetworkServiceAccountId() const NN_NOEXCEPT;
    void GetClientProfile(UserMigrationClientProfile* pOut) const NN_NOEXCEPT;
    void GetClientInfoForTransfer(ClientInfoForTransfer* pOut) const NN_NOEXCEPT;
    account::Uid GetUid() const NN_NOEXCEPT;

    // 移行リストの登録
    Result BeginImportMigrationList(const detail::MigrationListHeader& listHeader) NN_NOEXCEPT;
    Result UpdateImportMigrationList(const void* part, size_t partSize) NN_NOEXCEPT;
    Result EndImportMigrationList() NN_NOEXCEPT;
    Result ResetProgressInfo(void* buffer, size_t bufferSize) NN_NOEXCEPT;

    // 移行リストの情報取得
    size_t ReadMigrationList(detail::DataInfo* pOut, size_t count, size_t offset) const NN_NOEXCEPT;
    size_t CalculateCurrentShortfallOfUserSpace(void* buffer, size_t bufferSize) const NN_NOEXCEPT;
    size_t GetCurrentRelatedApplications(ncm::ApplicationId* pOut, size_t count) const NN_NOEXCEPT;

    // 進捗管理
    TransferInfo GetTotalTransferInfo() const NN_NOEXCEPT;
    TransferInfo GetCurrentTransferInfo() const NN_NOEXCEPT;
    void UpdateProgress(const detail::DataInfo& info) NN_NOEXCEPT;

    Result AdjustCurrentTransferInfo() NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::user
