﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/mem.h>
#include <nn/os.h>

namespace nn { namespace migration { namespace idc {
class SharedBufferConnection;
}}} // ~namespace nn::migration::idc

namespace nn { namespace migration { namespace idc {

/**
* @brief 共有バッファを管理し接続を作成するクラス。
*/
class SharedBufferConnectionManager
{
    NN_DISALLOW_MOVE(SharedBufferConnectionManager);
    NN_DISALLOW_COPY(SharedBufferConnectionManager);

public:
    /**
    * @brief 共有バッファのメモリ解放を行うオブジェクト。
    */
    class Deallocator
    {
    public:
        explicit Deallocator(mem::StandardAllocator* allocator) NN_NOEXCEPT
            : m_pAllocator(allocator)
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);
        }
        void Deallocate(void* addr) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pAllocator);
            m_pAllocator->Free(addr);
        }
    private:
        mem::StandardAllocator* m_pAllocator;
    };

    /**
    * @brief 送受信に用いる共有バッファ。
    */
    class SharedBuffer
    {
        NN_DISALLOW_COPY(SharedBuffer);
        NN_DISALLOW_MOVE(SharedBuffer);
    public:
        SharedBuffer() NN_NOEXCEPT;
        ~SharedBuffer() NN_NOEXCEPT;
        void Initialize(void* buffer, size_t size, Deallocator* pDeallocator) NN_NOEXCEPT;
        void Reset() NN_NOEXCEPT;

        bool WaitPushable(int timeoutMilliSeconds) NN_NOEXCEPT;
        bool WaitPoppable(int timeoutMilliSeconds) NN_NOEXCEPT;
        size_t Push(const void* stream, size_t size) NN_NOEXCEPT;
        size_t Pop(void* pOutStream, size_t outStreamSize) NN_NOEXCEPT;
        void BeginPush() NN_NOEXCEPT;
        void BeginPop() NN_NOEXCEPT;
        void EndPush() NN_NOEXCEPT;
        void EndPop() NN_NOEXCEPT;
        bool IsPushOpen() const NN_NOEXCEPT;
        bool IsPopOpen() const NN_NOEXCEPT;

    private:
        // timeoutMilliSeconds ミリ秒 wait() が true であれば true を返す。wait() が false になったら false を返す。
        bool TimedWaitWhileImpl(bool(SharedBuffer::*wait)() const, os::ConditionVariable& conditionVariable, int timeoutMilliSeconds) NN_NOEXCEPT;
        bool IsFullUnsafe() const NN_NOEXCEPT;
        bool IsEmptyUnsafe() const NN_NOEXCEPT;

        void*                   m_Buffer;
        size_t                  m_BufferSize;
        size_t                  m_UsedBufferSize;
        os::Mutex               m_BufferMutex;
        os::ConditionVariable   m_PushCondition;
        os::ConditionVariable   m_PopCondition;
        std::atomic<bool>       m_IsPushOpen;
        std::atomic<bool>       m_IsPopOpen;
        Deallocator*            m_pDeallocator;
    };

    // mem::StandardAllocator を使用して通信用のバッファを管理しているので bufferSize は管理領域の 16KB + 通信に使用するバッファのサイズ分以上必要。詳細は nn::mem::StandardAllocator::Initialize 参照。
    SharedBufferConnectionManager(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    ~SharedBufferConnectionManager() NN_NOEXCEPT;

    // 送信用と受信用それぞれに bufferSize のバッファを確保するので bufferSize * 2 以上のメモリを消費する。
    bool CreateConnection(SharedBufferConnection* pOutServer, SharedBufferConnection* pOutClient, size_t bufferSize) NN_NOEXCEPT;
    // 送信用と受信用それぞれに bufferSize のバッファを確保するので bufferSize * 2 以上のメモリを消費する（サーバー生成時点で両方の分を消費）。
    bool CreateServer(SharedBufferConnection* pOutServer, int* pOutServerId, size_t bufferSize) NN_NOEXCEPT;
    // サーバー側で明示的に接続待ちするための関数。既に接続済みの場合を含めて時間内に接続に成功した場合に true が返る。サーバー側がこの関数を呼んでいなくても Connect() は成功する。
    bool WaitConnection(int serverId, int timeoutSeconds) NN_NOEXCEPT;
    // 作成済みのサーバーに繋ぎにいく。
    bool Connect(SharedBufferConnection* pOut, int serverId) NN_NOEXCEPT;
    // 作成済みのサーバーもしくはクライアントを閉じる。両端が閉じられた際にメモリが開放される。SharedBufferConnection から呼ばれる。
    void Close(SharedBufferConnection* pConnection) NN_NOEXCEPT;
private:

    struct Connection
    {
        Connection() NN_NOEXCEPT;

        void Reset() NN_NOEXCEPT;
        void OpenServerEnd(int id) NN_NOEXCEPT;
        void OpenClientEnd(int id) NN_NOEXCEPT;
        void CloseServerEnd() NN_NOEXCEPT;
        void CloseClientEnd() NN_NOEXCEPT;

        int serverId;
        int clientId;
        SharedBuffer serverSendBuffer;
        SharedBuffer serverReceiveBuffer;
    };

    class IdGenerator
    {
    public:
        IdGenerator() NN_NOEXCEPT
            : m_Id(0) {}
        int Generate() NN_NOEXCEPT
        {
            return m_Id.fetch_add(1);
        }
    private:
        std::atomic<int> m_Id;
    } m_IdGenerator;

    std::pair<bool, Connection*> FindConnectionUnsafe(int id) NN_NOEXCEPT;
    Connection* AllocateConnectionUnsafe(size_t bufferSize) NN_NOEXCEPT;

    static const int MaxConnectionCount = 3; // TORIAEZU
    Connection m_ConnectionPool[MaxConnectionCount];
    mutable os::Mutex m_Mutex;
    mutable os::ConditionVariable m_ConditionVariable;
    mem::StandardAllocator m_Allocator;
    Deallocator m_Deallocator;
};

}}}

