﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/migration/detail/migration_AsyncExecutionResource.h>
#include <nn/migration/idc/migration_CommunicationApi.h>
#include <nn/migration/idc/migration_ServerContext.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace migration { namespace idc {

class Server
{
private:
    // Error レスポンスを返すためにクライアントからの User リクエストを読み捨てる Context。
    // クライアント側を受信可能な状態にし、MessageEncryptor のシーケンス番号を揃えるために受信・復号を最後まで行う。
    class UserRequestContextForErrorResponse
    {
    public:
        UserRequestContextForErrorResponse() NN_NOEXCEPT
            : m_CommandSize(0)
            , m_ConsumedSize(0)
        {
        }
        Result Consume(void*, size_t size) NN_NOEXCEPT
        {
            m_ConsumedSize += size;
            if( m_ConsumedSize < m_CommandSize )
            {
                NN_RESULT_THROW(ResultConsumeCommandContinue());
            }
            NN_SDK_ASSERT_EQUAL(m_ConsumedSize, m_CommandSize);
            NN_RESULT_SUCCESS;
        }
        void SetByteSizeToConsume(size_t size) NN_NOEXCEPT
        {
            m_CommandSize = size;
            m_ConsumedSize = 0;
        }
    private:
        size_t m_CommandSize;
        size_t m_ConsumedSize;
    };
public:

    template <typename ServerContextType>
    struct WaitUserCommandBuffer
    {
        Bit8 communicationBuffer[ServerContextType::EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize];
        UserCommandMediatorWorkBuffer<typename ServerContextType::EncryptionPolicy> userCommandMediatorWorkBuffer;
    };

    template <typename ServerContextType>
    struct WaitUserCommandBufferForThreaded
    {
        Bit8 communicationBuffer[ServerContextType::EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize];
        ThreadedUserCommandMediatorWorkBuffer<typename ServerContextType::EncryptionPolicy> userCommandMediatorWorkBuffer;
    };


    template <typename ConnectionType, typename ServerContextType>
    static Result WaitConnection(const ConnectionType& connection, ServerContextType& serverContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!serverContext.IsConnected());
        while( !serverContext.IsConnected() )
        {
            NN_RESULT_DO(Receive<>(connection, serverContext.GetConsumerRef(), timeoutSeconds, workBuffer, workBufferSize, pCancellable));
            NN_RESULT_DO(Send<>(connection, serverContext.GetProducerRef(), timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        }
        NN_RESULT_SUCCESS;
    }

    template <typename ConnectionType, typename ServerContextType>
    static Result WaitTermination(const ConnectionType& connection, ServerContextType& serverContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(serverContext.IsConnected());
        while( serverContext.IsConnected() )
        {
            NN_RESULT_DO(Receive<>(connection, serverContext.GetConsumerRef(), timeoutSeconds, workBuffer, workBufferSize, pCancellable));
            NN_RESULT_DO(Send<>(connection, serverContext.GetProducerRef(), timeoutSeconds, workBuffer, workBufferSize, pCancellable));
        }
        NN_RESULT_SUCCESS;
    }

    // 受信/消費を並列化していない。
    template <typename ConnectionType, typename ServerContextType, typename UserContextType>
    static Result WaitUserCommand(const ConnectionType& connection, ServerContextType& serverContext, UserContextType& userContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
        migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::alignment_of<WaitUserCommandBuffer<ServerContextType>>::value == 1u);
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(WaitUserCommandBuffer<ServerContextType>));
        NN_UNUSED(workBufferSize);
        NN_SDK_REQUIRES(serverContext.IsConnected());
        NN_SDK_REQUIRES(!serverContext.HasErrorResponse());

        auto pBuffer = reinterpret_cast<WaitUserCommandBuffer<ServerContextType>*>(workBuffer);
        {
            auto holder = serverContext.AcquireConsumerHolder(
                &userContext, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Receive<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        {
            auto holder = serverContext.AcquireProducerHolder(
                &userContext, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Send<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        NN_RESULT_SUCCESS;
    }

    // 受信/消費を並列化していない。
    template <typename ConnectionType, typename ServerContextType>
    static Result WaitUserCommandAndRespondWithError(const ConnectionType& connection, ServerContextType& serverContext, int timeoutSeconds,
        void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
        migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::alignment_of<WaitUserCommandBuffer<ServerContextType>>::value == 1u);
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(WaitUserCommandBuffer<ServerContextType>));
        NN_UNUSED(workBufferSize);
        NN_SDK_REQUIRES(serverContext.IsConnected());
        NN_SDK_REQUIRES(serverContext.HasErrorResponse());

        auto pBuffer = reinterpret_cast<WaitUserCommandBuffer<ServerContextType>*>(workBuffer);
        {
            UserRequestContextForErrorResponse contextForErrorResponse;
            auto holder = serverContext.AcquireConsumerHolder(
                &contextForErrorResponse, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Receive<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        {
            NN_RESULT_DO(Send<>(
                connection, serverContext.GetConsumerRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        NN_RESULT_SUCCESS;
    }

    // 受信/消費を並列化。
    template <typename ConnectionType, typename ServerContextType, typename UserContextType>
    static Result WaitUserCommand(const ConnectionType& connection, ServerContextType& serverContext, UserContextType& userContext, int timeoutSeconds,
        migration::detail::ThreadResourceType* pThreadResource, void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
        migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::alignment_of<WaitUserCommandBufferForThreaded<ServerContextType>>::value == 1u);
        NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);
        NN_SDK_REQUIRES_NOT_NULL(pThreadResource);
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(WaitUserCommandBufferForThreaded<ServerContextType>));
        NN_UNUSED(workBufferSize);
        NN_SDK_REQUIRES(serverContext.IsConnected());
        auto pBuffer = reinterpret_cast<WaitUserCommandBufferForThreaded<ServerContextType>*>(workBuffer);
        {
            auto holder = serverContext.AcquireThreadedConsumerHolder(&userContext, pThreadResource, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Receive<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        {
            auto holder = serverContext.AcquireThreadedProducerHolder(&userContext, pThreadResource, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Send<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        NN_RESULT_SUCCESS;
    }

    // 受信/消費を並列化。
    template <typename ConnectionType, typename ServerContextType>
    static Result WaitUserCommandAndRespondWithError(const ConnectionType& connection, ServerContextType& serverContext, int timeoutSeconds,
        migration::detail::ThreadResourceType* pThreadResource, void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
        migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(std::alignment_of<WaitUserCommandBufferForThreaded<ServerContextType>>::value == 1u);
        NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);
        NN_SDK_REQUIRES_NOT_NULL(pThreadResource);
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(WaitUserCommandBufferForThreaded<ServerContextType>));
        NN_UNUSED(workBufferSize);
        NN_SDK_REQUIRES(serverContext.IsConnected());
        auto pBuffer = reinterpret_cast<WaitUserCommandBufferForThreaded<ServerContextType>*>(workBuffer);
        {
            UserRequestContextForErrorResponse contextForErrorResponse;
            auto holder = serverContext.AcquireThreadedConsumerHolder(&contextForErrorResponse, pThreadResource, &pBuffer->userCommandMediatorWorkBuffer);
            NN_RESULT_DO(Receive<>(
                connection, holder.GetProcessorRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        {
            NN_RESULT_DO(Send<>(
                connection, serverContext.GetConsumerRef(), timeoutSeconds,
                pBuffer->communicationBuffer, sizeof(pBuffer->communicationBuffer), pCancellable, pSpeedMonitor));
        }
        NN_RESULT_SUCCESS;
    }
};

}}}
