﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/migration/idc/detail/migration_Result.h>
#include <nn/migration/idc/detail/migration_Util.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>

namespace nn { namespace migration { namespace idc {

inline int GetRemainingTimeoutSeconds(int timeoutSeconds, const os::Tick& startTick)
{
    return timeoutSeconds > 0 ? static_cast<int>(timeoutSeconds - (os::GetSystemTick() - startTick).ToTimeSpan().GetSeconds()) : 0;
}

template <typename ConnectionType, typename ProducerType>
Result Send(
    const ConnectionType& connection, ProducerType& producer, int timeoutSeconds,
    void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
    migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER(workBufferSize, 0u);

    auto startTick = os::GetSystemTick();
    bool done = false;

    while( !done )
    {
        NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), ResultCanceled());
        NN_RESULT_THROW_UNLESS(timeoutSeconds == 0 || (os::GetSystemTick() - startTick).ToTimeSpan() < TimeSpan::FromSeconds(timeoutSeconds), ResultTimeout());

        size_t producedSize;
        auto result = producer.Produce(&producedSize, workBuffer, workBufferSize);
        if( ResultProduceCommandPause::Includes(result) )
        {
            NN_SDK_ASSERT_EQUAL(producedSize, 0u);
            os::SleepThread(nn::TimeSpan::FromSeconds(1));
            continue;
        }
        NN_RESULT_THROW_UNLESS(result.IsSuccess() || ResultProduceCommandContinue::Includes(result), result);
        NN_SDK_ASSERT_GREATER(producedSize, 0u);
        done = result.IsSuccess();

        size_t totalSentSize = 0;
        while( totalSentSize < producedSize )
        {
            auto sendTimeoutSeconds = GetRemainingTimeoutSeconds(timeoutSeconds, startTick);
            NN_RESULT_THROW_UNLESS(timeoutSeconds == 0 || sendTimeoutSeconds > 0, ResultTimeout());
            size_t sentSize;
            NN_RESULT_DO(connection.Send(&sentSize, reinterpret_cast<Bit8*>(workBuffer) + totalSentSize, producedSize - totalSentSize, sendTimeoutSeconds, pCancellable, pSpeedMonitor));
            totalSentSize += sentSize;
        }
        NN_SDK_ASSERT_EQUAL(totalSentSize, producedSize);
    }
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename ConsumerType>
Result Receive(
    const ConnectionType& connection, ConsumerType& consumer, int timeoutSeconds,
    void* workBuffer, size_t workBufferSize, const migration::detail::Cancellable* pCancellable,
    migration::idc::detail::TransferSpeedMonitor* pSpeedMonitor = nullptr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER(workBufferSize, 0u);

    auto startTick = os::GetSystemTick();

    while( NN_STATIC_CONDITION(true) )
    {
        auto recvTimeoutSeconds = GetRemainingTimeoutSeconds(timeoutSeconds, startTick);
        NN_RESULT_THROW_UNLESS(timeoutSeconds == 0 || recvTimeoutSeconds > 0, ResultTimeout());
        size_t recvSize;
        NN_RESULT_DO(connection.Receive(&recvSize, reinterpret_cast<Bit8*>(workBuffer), workBufferSize, recvTimeoutSeconds, pCancellable, pSpeedMonitor));

RetryConsume:

        NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), ResultCanceled());
        NN_RESULT_THROW_UNLESS(timeoutSeconds == 0 || (os::GetSystemTick() - startTick).ToTimeSpan() < TimeSpan::FromSeconds(timeoutSeconds), ResultTimeout());

        auto result = consumer.Consume(workBuffer, recvSize);
        NN_RESULT_THROW_UNLESS(result.IsSuccess() || ResultConsumeCommandPause::Includes(result) || ResultConsumeCommandContinue::Includes(result), result);
        if( ResultConsumeCommandPause::Includes(result) )
        {
            os::SleepThread(nn::TimeSpan::FromSeconds(1));
            goto RetryConsume;
        }
        else if( ResultConsumeCommandContinue::Includes(result) )
        {
            continue;
        }
        NN_SDK_ASSERT(result.IsSuccess());
        NN_RESULT_SUCCESS;
    }
}

}}}
