﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/crypto.h>
#include <nn/migration/idc/migration_MessageEncryptor.h>

namespace nn { namespace migration { namespace idc {

struct KeyExchangeCommandConfig
{
    static const size_t ChallengeSize   = 32; // 通信プロトコル暗号鍵の交換時に用いる検証用のバイト列のサイズ。
    static const size_t SaltSize        = 32; // Challengeに付与する salt のサイズ。
    static const size_t Iv0Size         = 16; // 移行先から移行元に鍵交換1のレスポンスを返す際の暗号化に用いる IV のサイズ。
    static const size_t PassphraseSize  = 40; // 通信プロトコル暗号鍵のサイズ。
    static const size_t KeySize         = 16; // 鍵交換時の暗号化に用いる鍵のサイズ。

    typedef Bit8 Challenge[KeyExchangeCommandConfig::ChallengeSize];
    typedef Bit8 Salt[SaltSize];
    typedef Bit8 Iv0[Iv0Size];
    typedef Bit8 Passphrase[PassphraseSize];
    typedef Bit8 Key[KeySize];
};

enum CommandKind : Bit8
{
    Initiate0   = 0x00, // 鍵交換1。「通信プロトコル暗号鍵の交換」の最初のコマンド。
    Initiate1   = 0x01, // 鍵交換2。「通信プロトコル暗号鍵の交換」の最後のコマンド。
    Resume0     = 0x08, // 暗号鍵を交換し、何らかの理由で通信路が断絶したあとの通信再開時の最初のコマンド。
    Resume1     = 0x09, // 暗号鍵を交換し、何らかの理由で通信路が断絶したあとの通信再開時の最後のコマンド。
    Terminate   = 0x10, // 通信を終了する際のコマンド。
    User        = 0x80, // アプリケーション定義の通信を行うためのコマンド。
    Error       = 0xFF, // サーバーでエラーを検知した場合に、その情報をクライアントに応答するためのコマンド。
};

enum ErrorKind : Bit8
{
    ServiceStopped = 0x00, // サービスの停止。
};

struct Initiate0Request
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[KeyExchangeCommandConfig::ChallengeSize];
    Bit8 mac[crypto::Aes128CmacGenerator::MacSize];
};

struct Initiate0Response
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 iv[KeyExchangeCommandConfig::Iv0Size];
    Bit8 data[KeyExchangeCommandConfig::PassphraseSize + KeyExchangeCommandConfig::ChallengeSize * 2 + 8];
    Bit8 mac[crypto::Aes128CmacGenerator::MacSize];
};
#if !defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013)
NN_STATIC_ASSERT(sizeof(Initiate0Response::data) % crypto::Aes128CtrEncryptor::BlockSize == 0);
#endif

struct Initiate1Request
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[KeyExchangeCommandConfig::ChallengeSize + MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct Initiate1Response
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct Resume0Request
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[KeyExchangeCommandConfig::ChallengeSize + MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct Resume0Response
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[KeyExchangeCommandConfig::ChallengeSize * 2 + MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct Resume1Request
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[KeyExchangeCommandConfig::ChallengeSize + MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct Resume1Response
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct TerminateRequest
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct TerminateResponse
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct ErrorInfo
{
    Bit8 errorId;
    Bit8 padding[7];
};

struct ErrorResponse
{
    Bit8 commandId;
    Bit8 data[sizeof(ErrorInfo) + MessageEncryptorConfig::SequenceManagementDataSize];
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

struct UserCommandHeader
{
    Bit8 commandId;
    Bit8 padding[7];
    Bit8 data[MessageEncryptorConfig::SequenceManagementDataSize + sizeof(uint64_t) + sizeof(uint64_t)]; // シーケンス管理データ + ブロック長 + データ長
    Bit8 mac[crypto::Aes128GcmEncryptor::MacSize];
};

typedef Initiate0Response MaxSizeResponse;
const size_t ResponseCommandSizeMax = sizeof(MaxSizeResponse);
NN_STATIC_ASSERT(sizeof(Initiate0Response) <= sizeof(MaxSizeResponse));
NN_STATIC_ASSERT(sizeof(Initiate1Response) <= sizeof(MaxSizeResponse));
NN_STATIC_ASSERT(sizeof(Resume0Response)   <= sizeof(MaxSizeResponse));
NN_STATIC_ASSERT(sizeof(Resume1Response)   <= sizeof(MaxSizeResponse));
NN_STATIC_ASSERT(sizeof(TerminateResponse) <= sizeof(MaxSizeResponse));
NN_STATIC_ASSERT(sizeof(UserCommandHeader) <= sizeof(MaxSizeResponse));
NN_STATIC_ASSERT(sizeof(ErrorResponse)     <= sizeof(MaxSizeResponse));

typedef Initiate1Request MaxSizeRequest;
const size_t ReqeustCommandSizeMax = sizeof(Initiate1Request);
NN_STATIC_ASSERT(sizeof(Initiate0Request)  <= sizeof(MaxSizeRequest));
NN_STATIC_ASSERT(sizeof(Initiate1Request)  <= sizeof(MaxSizeRequest));
NN_STATIC_ASSERT(sizeof(Resume0Request)    <= sizeof(MaxSizeRequest));
NN_STATIC_ASSERT(sizeof(Resume1Request)    <= sizeof(MaxSizeRequest));
NN_STATIC_ASSERT(sizeof(TerminateRequest)  <= sizeof(MaxSizeRequest));
NN_STATIC_ASSERT(sizeof(UserCommandHeader) <= sizeof(MaxSizeRequest));

}}}
