﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <limits>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

namespace nn {namespace migration { namespace detail {
template <typename T>
class SharedResource;

template <typename T>
class UniqueResource;
}}}
namespace nn { namespace migration { namespace detail {

class Counter
{
    template<typename T>
    friend class SharedResource;
    template<typename T>
    friend class UniqueResource;

private:
    std::atomic<uint32_t> value;
    void Add() NN_NOEXCEPT
    {
        auto v = value.fetch_add(1);
        NN_ABORT_UNLESS(v < std::numeric_limits<uint32_t>::max());
    }
    void Sub() NN_NOEXCEPT
    {
        auto v = value.fetch_sub(1);
        NN_ABORT_UNLESS(v > 0);
    }
public:
    Counter() NN_NOEXCEPT
        : value(0)
    {
    }
    uint32_t Get() const NN_NOEXCEPT
    {
        return value.load();
    }
};

template <typename Resource>
class SharedResource
{
private:
    Resource* m_pResource;
    Counter* m_pCounter;
    void Swap(SharedResource<Resource>& rhs) NN_NOEXCEPT
    {
        std::swap(this->m_pResource, rhs.m_pResource);
        std::swap(this->m_pCounter, rhs.m_pCounter);
    }

    typedef void (SharedResource<Resource>::*BoolType)() const;
    void BoolTypeHelper() const NN_NOEXCEPT {}

public:
    SharedResource() NN_NOEXCEPT
        : m_pResource(nullptr)
        , m_pCounter(nullptr)
    {
    }
    SharedResource(Resource& resource, Counter& counter) NN_NOEXCEPT
        : m_pResource(&resource)
        , m_pCounter(&counter)
    {
        m_pCounter->Add();
    }
    SharedResource(const SharedResource<Resource>& rhs) NN_NOEXCEPT
        : m_pResource(rhs.m_pResource)
        , m_pCounter(rhs.m_pCounter)
    {
        if (m_pCounter)
        {
            m_pCounter->Add();
        }
    }
    SharedResource(SharedResource<Resource>&& rhs) NN_NOEXCEPT
        : m_pResource(rhs.m_pResource)
        , m_pCounter(rhs.m_pCounter)
    {
        rhs.m_pResource = nullptr;
        rhs.m_pCounter = nullptr;
    }
    ~SharedResource() NN_NOEXCEPT
    {
        if (m_pCounter)
        {
            m_pCounter->Sub();
        }
    }
    SharedResource<Resource>& operator=(const SharedResource<Resource>& rhs) NN_NOEXCEPT
    {
        SharedResource tmp(rhs);
        tmp.Swap(*this);
        return *this;
    }
    SharedResource<Resource>& operator=(SharedResource<Resource>&& rhs) NN_NOEXCEPT
    {
        SharedResource<Resource> tmp(std::move(rhs));
        tmp.Swap(*this);
        return *this;
    }
    NN_IMPLICIT operator BoolType() const NN_NOEXCEPT
    {
        return m_pResource != nullptr
            ? &SharedResource::BoolTypeHelper
            : nullptr;
    }
    Resource& Get() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pResource);
        return *m_pResource;
    }
};

template <typename Resource>
class UniqueResource
{
private:
    Resource* m_pResource;
    Counter* m_pCounter;
    void Swap(UniqueResource<Resource>& rhs) NN_NOEXCEPT
    {
        std::swap(this->m_pResource, rhs.m_pResource);
        std::swap(this->m_pCounter, rhs.m_pCounter);
    }

    typedef void (UniqueResource<Resource>::*BoolType)() const;
    void BoolTypeHelper() const NN_NOEXCEPT {}

public:
    UniqueResource() NN_NOEXCEPT
        : m_pResource(nullptr)
        , m_pCounter(nullptr)
    {
    }
    UniqueResource(Resource& resource, Counter& counter) NN_NOEXCEPT
        : m_pResource(&resource)
        , m_pCounter(&counter)
    {
        m_pCounter->Add();
    }
    UniqueResource(UniqueResource<Resource>&& rhs) NN_NOEXCEPT
        : m_pResource(rhs.m_pResource)
        , m_pCounter(rhs.m_pCounter)
    {
        rhs.m_pResource = nullptr;
        rhs.m_pCounter = nullptr;
    }
    ~UniqueResource() NN_NOEXCEPT
    {
        if (m_pCounter)
        {
            m_pCounter->Sub();
        }
    }
    UniqueResource<Resource>& operator=(UniqueResource<Resource>&& rhs) NN_NOEXCEPT
    {
        UniqueResource<Resource> tmp(std::move(rhs));
        tmp.Swap(*this);
        return *this;
    }
    NN_IMPLICIT operator BoolType() const NN_NOEXCEPT
    {
        return m_pResource != nullptr
            ? &UniqueResource::BoolTypeHelper
            : nullptr;
    }
    Resource& Get() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pResource);
        return *m_pResource;
    }
};

}}} // ~namespace nn::migration::detail
