﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/account/account_ApiForMigrationService.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_SharedResource.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_MemoryHeapCommon.h>

namespace nn { namespace migration { namespace detail {

Result ClearUserData(const account::Uid& uid, const Cancellable* pCancellable) NN_NOEXCEPT;

class LoginSessionBase
{
public:
    virtual Result GetNetworkServiceAccountId(account::NetworkServiceAccountId* pOut) NN_NOEXCEPT = 0;
    virtual Result AcquireNetworkServiceAccountIdToken(
        size_t* pOut, char* buffer, size_t bufferSize,
        const account::SystemProgramIdentification& sysAppId, const detail::Cancellable* pCancellable) NN_NOEXCEPT = 0;
};

class LocalUserLoginSession
    : public LoginSessionBase
{
    NN_DISALLOW_COPY(LocalUserLoginSession);
    NN_DISALLOW_MOVE(LocalUserLoginSession);

private:
    account::Uid m_Uid;
    bool m_Initialized;

public:
    LocalUserLoginSession() NN_NOEXCEPT;
    void Initialize(const account::Uid& uid) NN_NOEXCEPT;

    virtual Result GetNetworkServiceAccountId(account::NetworkServiceAccountId* pOut) NN_NOEXCEPT final NN_OVERRIDE;

    virtual Result AcquireNetworkServiceAccountIdToken(
        size_t* pOut, char* buffer, size_t bufferSize,
        const account::SystemProgramIdentification& sysAppId, const detail::Cancellable* pCancellable) NN_NOEXCEPT final NN_OVERRIDE;

    Result DeleteUser(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

class RemoteUserLoginSession
    : public LoginSessionBase
{
    NN_DISALLOW_COPY(RemoteUserLoginSession);
    NN_DISALLOW_MOVE(RemoteUserLoginSession);

public:
    typedef std::aligned_storage<account::RequiredBufferSizeForExternalNetworkServiceAccountRegistrar, os::MemoryPageSize>::type WorkMemory;

private:
    WorkMemory* m_Buffer;
    account::ExternalNetworkServiceAccountRegistrar m_Registrar;

    mutable os::Mutex m_Lock;
    bool m_Initialized;
    account::Uid m_Uid;
    bool m_IsRegistered;

public:
    RemoteUserLoginSession(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    account::SessionId Create() NN_NOEXCEPT;

    virtual Result GetNetworkServiceAccountId(account::NetworkServiceAccountId* pOut) NN_NOEXCEPT final NN_OVERRIDE;
    Result GetNickname(account::Nickname* pOut) NN_NOEXCEPT;
    Result GetProfileImage(size_t* pOut, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    virtual Result AcquireNetworkServiceAccountIdToken(
        size_t* pOut, char* buffer, size_t bufferSize,
        const account::SystemProgramIdentification& sysAppId, const detail::Cancellable* pCancellable) NN_NOEXCEPT final NN_OVERRIDE;

    void SetUid(const account::Uid& uid) NN_NOEXCEPT;
    void RegisterUser() NN_NOEXCEPT;
    Result RegisterNetworkServiceAccount(const detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

class ClientLoginSessionHolder
{
private:
    RemoteUserLoginSession m_Session;
    os::Mutex m_Lock;
    Counter m_Counter;

public:
    ClientLoginSessionHolder(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    ~ClientLoginSessionHolder() NN_NOEXCEPT;
    UniqueResource<RemoteUserLoginSession> Acquire() NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::detail
