﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/migration/detail/migration_MigrationList.h>
#include <nn/migration/detail/migration_Settings.h>
#include <nn/migration/idc/migration_Server.h>
#include <nn/migration/idc/migration_ServerContext.h>
#include <nn/migration/idc/migration_UserCommandEncryptor.h>

namespace nn { namespace migration { namespace detail {

class IdcServerBase
{
private:
    struct Resource
    {
        FsKeySeedDigest fsKeySeedDigest;
        OnetimeToken ott;
    };

    fs::SaveDataTransferManager m_FsManager;
    Resource m_Resource;

protected:
    typedef Authenticator::WorkBuffer AuthenticationBuffer;

    Result InitializeFileSystemParameters(LoginSessionBase& login, AuthenticationBuffer& authBuffer, const Cancellable* pCancellable) NN_NOEXCEPT;
    const OnetimeToken& GetOnetimeToken() NN_NOEXCEPT
    {
        return m_Resource.ott;
    }
    const FsKeySeedDigest& GetFsKeySeedDigest() NN_NOEXCEPT
    {
        return m_Resource.fsKeySeedDigest;
    }

public:
    Result OpenSaveDataExporter(std::unique_ptr<fs::SaveDataExporter>* outValue, const DataInfo& dataInfo) NN_NOEXCEPT;
};

template <typename ConnectionType, typename EncryptionPolicy>
class IdcServer
    : public IdcServerBase
{
private:
    typedef idc::ServerContext<EncryptionPolicy> IdcContext;
    struct Resource
    {
        union
        {
            AuthenticationBuffer authenticationBuffer;
            char communicationBuffer[sizeof(idc::Server::WaitUserCommandBufferForThreaded<IdcContext>)];
        } u;

        NN_STATIC_ASSERT(sizeof(idc::Server::WaitUserCommandBufferForThreaded<IdcContext>) >= sizeof(idc::Server::WaitUserCommandBuffer<IdcContext>));
    };

    IdcContext m_IdcContext;
    Resource m_Resource;
    ThreadResource m_ThreadResource;

protected:
    template <typename UserContext>
    Result WaitUserCommand(const ConnectionType& connection, UserContext& userContext, int timeout, const Cancellable* pCancellable) NN_NOEXCEPT
    {
        idc::detail::TransferSpeedMonitor speedMonitor(1, nn::TimeSpan::FromSeconds(detail::GetTransferLowSpeedTimeoutSeconds()));
        return idc::Server::WaitUserCommand(
            connection, m_IdcContext, userContext, timeout, &m_ThreadResource.Get(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable, &speedMonitor);
    }

public:
    explicit IdcServer(ThreadResource&& threadResource) NN_NOEXCEPT
        : m_ThreadResource(std::move(threadResource))
    {
    }
    Result Initialize(LoginSessionBase& login, const Cancellable* pCancellable) NN_NOEXCEPT;
    Result HandleOnetimeTokenRequest(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT;
    Result HandleFsKeySeedVerificationRequest(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::detail

#include <nn/migration/detail/migration_IdcImpl.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace detail {

template <typename ConnectionType, typename EncryptionPolicy>
inline Result IdcServer<ConnectionType, EncryptionPolicy>::Initialize(LoginSessionBase& login, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[IdcServer] Acquiring FsKeySeed from authentication server\n");
    NN_RESULT_DO(InitializeFileSystemParameters(login, m_Resource.u.authenticationBuffer, pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
inline Result IdcServer<ConnectionType, EncryptionPolicy>::HandleOnetimeTokenRequest(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[IdcServer] Waiting for idc connection\n");
    NN_RESULT_DO(idc::Server::WaitConnection(
        connection, m_IdcContext, detail::GetTransferMessagingTimeoutSeconds(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable));

    GetAuthenticationCodeServerApi api(GetOnetimeToken());
    NN_RESULT_DO(WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));

    NN_MIGRATION_DETAIL_TRACE("[IdcServer] Waiting for idc termination\n");
    NN_RESULT_DO(idc::Server::WaitTermination(
        connection, m_IdcContext, detail::GetTransferMessagingTimeoutSeconds(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
inline Result IdcServer<ConnectionType, EncryptionPolicy>::HandleFsKeySeedVerificationRequest(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[IdcServer] Waiting for idc re-connection\n");
    NN_RESULT_DO(idc::Server::WaitConnection(
        connection, m_IdcContext, detail::GetTransferMessagingTimeoutSeconds(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable));

    VerifyAuthenticationDigestServerApi api(GetFsKeySeedDigest());
    NN_RESULT_DO(WaitUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    return api.GetResult();
}

}}} // ~namespace nn::migration::detail
