﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/migration/detail/migration_MigrationList.h>
#include <nn/migration/detail/migration_Settings.h>
#include <nn/migration/idc/migration_Client.h>
#include <nn/migration/idc/migration_ClientContext.h>
#include <nn/migration/idc/migration_UserCommandEncryptor.h>
#include <nn/migration/idc/detail/migration_Util.h>

namespace nn { namespace migration { namespace detail {

class IdcClientBase
{
private:
    struct Resource
    {
        FsKeySeedDigest fsKeySeedDigest;
    };

    LoginSessionBase* m_pLogin;
    fs::SaveDataTransferManager m_FsManager;
    Resource m_Resource;

protected:
    typedef Authenticator::WorkBuffer AuthenticationBuffer;

    IdcClientBase() NN_NOEXCEPT
        : m_pLogin(nullptr)
    {
    }
    void Initialize(LoginSessionBase& login) NN_NOEXCEPT
    {
        m_pLogin = &login;
    }
    Result InitializeFileSystemParameters(const OnetimeToken& ott, AuthenticationBuffer& authBuffer, const Cancellable* pCancellable) NN_NOEXCEPT;
    const FsKeySeedDigest& GetFsKeySeedDigest() NN_NOEXCEPT
    {
        return m_Resource.fsKeySeedDigest;
    }

public:
    Result OpenSaveDataImporter(std::unique_ptr<fs::SaveDataImporter>* outValue, const account::Uid& uid, const DataInfo& dataInfo, const void* exportInitialData, size_t exportInitialDataSize) NN_NOEXCEPT;
};

template <typename ConnectionType, typename EncryptionPolicy>
class IdcClient
    : public IdcClientBase
{
    NN_STATIC_ASSERT(EncryptionPolicy::UserCommandBlockSize >= 256);

private:
    typedef idc::ClientContext<EncryptionPolicy> IdcContext;
    struct Resource
    {
        OnetimeToken ott;
        union
        {
            AuthenticationBuffer authenticationBuffer;
            char communicationBuffer[sizeof(idc::Client::InvokeUserCommandBufferForThreaded<IdcContext>)];
        } u;

        NN_STATIC_ASSERT(sizeof(idc::Client::InvokeUserCommandBufferForThreaded<IdcContext>) >= sizeof(idc::Client::InvokeUserCommandBuffer<IdcContext>));
    };

    IdcContext m_IdcContext;
    Resource m_Resource;
    ThreadResource m_ThreadResource;

protected:
    template <typename UserContext>
    Result InvokeUserCommand(const ConnectionType& connection, UserContext& userContext, int timeout, const Cancellable* pCancellable) NN_NOEXCEPT
    {
        idc::detail::TransferSpeedMonitor speedMonitor(1, nn::TimeSpan::FromSeconds(detail::GetTransferLowSpeedTimeoutSeconds()));
        return idc::Client::InvokeUserCommand(
            connection, m_IdcContext, userContext, timeout, &m_ThreadResource.Get(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable, &speedMonitor);
    }

public:
    explicit IdcClient(ThreadResource&& threadResource) NN_NOEXCEPT
        : m_ThreadResource(std::move(threadResource))
    {
    }
    Result Initialize(LoginSessionBase& login, const Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetOnetimeToken(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT;
    Result AuthenticateWithOnetimeToken(const Cancellable* pCancellable) NN_NOEXCEPT;
    Result VerifyFsKeySeed(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::detail

#include <nn/migration/detail/migration_IdcImpl.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace detail {

template <typename ConnectionType, typename EncryptionPolicy>
inline Result IdcClient<ConnectionType, EncryptionPolicy>::Initialize(LoginSessionBase& login, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_UNUSED(pCancellable);
    IdcClientBase::Initialize(login);
    NN_RESULT_SUCCESS;
}


template <typename ConnectionType, typename EncryptionPolicy>
inline Result IdcClient<ConnectionType, EncryptionPolicy>::GetOnetimeToken(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[IdcClient] Initiating idc connection\n");
    NN_RESULT_DO(idc::Client::Initiate(
        connection, m_IdcContext, detail::GetTransferMessagingTimeoutSeconds(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable));

    // OTT 受信
    GetAuthenticationCodeClientApi api;
    NN_RESULT_DO(InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    api.GetOnetimeToken(&m_Resource.ott);

    // クライアントはインターネットに接続するため、一度切断する
    NN_MIGRATION_DETAIL_TRACE("[IdcClient] Terminating idc connection\n");
    NN_RESULT_DO(idc::Client::Terminate(
        connection, m_IdcContext, detail::GetTransferMessagingTimeoutSeconds(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
inline Result IdcClient<ConnectionType, EncryptionPolicy>::AuthenticateWithOnetimeToken(const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[IdcClient] Acquiring FsKeySeed from authentication server\n");
    NN_RESULT_DO(InitializeFileSystemParameters(m_Resource.ott, m_Resource.u.authenticationBuffer, pCancellable));
    NN_RESULT_SUCCESS;
}

template <typename ConnectionType, typename EncryptionPolicy>
inline Result IdcClient<ConnectionType, EncryptionPolicy>::VerifyFsKeySeed(const ConnectionType& connection, const Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_TRACE("[IdcClient] Resuming idc connection\n");
    NN_RESULT_DO(idc::Client::Resume(
        connection, m_IdcContext, detail::GetTransferMessagingTimeoutSeconds(), m_Resource.u.communicationBuffer, sizeof(m_Resource.u.communicationBuffer), pCancellable));

    // FsKeySeedDigest の検証
    VerifyAuthenticationDigestClientApi api(GetFsKeySeedDigest());
    NN_RESULT_DO(InvokeUserCommand(connection, api, detail::GetTransferMessagingTimeoutSeconds(), pCancellable));
    return api.GetResult();
}
}}} // ~namespace nn::migration::detail

