﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/migration/detail/migration_ThreadResourceAllocator.h>
#include <nn/os/os_ConditionVariable.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_ThreadTypes.h>

namespace nn { namespace migration { namespace detail {


class AsyncExecutionResource
{
private:
    os::ThreadType m_Thread;

    struct Task
    {
        bool _isAttached;
        void(* _f)(void*);
        void* _data;
        os::SystemEventType* _pEvent;

        void Attach(void(*f)(void*), void* data, os::SystemEventType* pEvent) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(!_isAttached);
            this->_f = f;
            this->_data = data;
            this->_pEvent = pEvent;
            _isAttached = true;
        }
        void Reset() NN_NOEXCEPT
        {
            // NN_SDK_ASSERT(_isAttached);
            this->_f = nullptr;
            this->_data = nullptr;
            this->_pEvent = nullptr;
            _isAttached = false;
        }
        void Invoke() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(_isAttached);
            _f(_data);
            os::SignalSystemEvent(_pEvent);
        }
    } m_Task;

    os::Event m_StopEvent;
    os::Event m_AttachEvent;
    os::ConditionVariable m_DetachCondition;
    os::Mutex m_Lock;

    ThreadResource m_Resource;

    static void ThreadFunction(void* pObj) NN_NOEXCEPT
    {
        reinterpret_cast<AsyncExecutionResource*>(pObj)->ThreadFunctionImpl();
    }

    void ThreadFunctionImpl() NN_NOEXCEPT;
    void Attach(void(*f)(void*), void* data, os::SystemEventType* pEvent) NN_NOEXCEPT;
    bool TryAttach(void(*f)(void*), void* data, os::SystemEventType* pEvent) NN_NOEXCEPT;

public:
    explicit AsyncExecutionResource(ThreadResource&& threadResource) NN_NOEXCEPT;
    ~AsyncExecutionResource() NN_NOEXCEPT;

    template <typename T>
    void Delegate(void(*function)(T*), T* data, os::SystemEvent& e) NN_NOEXCEPT
    {
        // この関数はサブスレッドに処理をアタッチ完了するまでブロックする
        NN_SDK_ASSERT(!e.TryWait());
        Attach(reinterpret_cast<void(*)(void*)>(function), data, e.GetBase());
    }

    template <typename T>
    bool TryDelegate(void(*function)(T*), T* data, os::SystemEvent& e) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!e.TryWait());
        return TryAttach(reinterpret_cast<void(*)(void*)>(function), data, e.GetBase());
    }
};

}}} // ~namespace nn::migration::detail
