﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>

#include <nn/nn_SdkLog.h>
#include <nn/migration/migration_Result.h>
#include <nn/migration/detail/migration_AsyncExecutionResource.h>
#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace migration { namespace detail {

template <typename Impl>
class AsyncContextImpl
    : public Cancellable
{

private:
    os::SystemEvent m_Event;
    Result m_Result;
    Impl m_Impl;

    static void Execute(AsyncContextImpl<Impl>* pObj) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!pObj->m_Event.TryWait());
        pObj->m_Result = !pObj->IsCanceled()
            ? pObj->m_Impl.Execute(static_cast<Cancellable*>(pObj))
            : ResultCanceled();
    }

public:
    template <typename... Args>
    AsyncContextImpl(AsyncExecutionResource& resource, Args&&... args) NN_NOEXCEPT
        : m_Event(os::EventClearMode_ManualClear, true)
        , m_Impl(std::forward<Args>(args)...)
    {
        resource.Delegate<>(Execute, this, m_Event);
    }
    ~AsyncContextImpl() NN_NOEXCEPT
    {
        if (!m_Event.TryWait())
        {
            Cancellable::Cancel();
            m_Event.Wait();
            NN_MIGRATION_DETAIL_WARN("AsyncContext canceled at ~AsyncContextImpl(): %03d-%04d (%08lx)\n", m_Result.GetModule(), m_Result.GetDescription(), m_Result.GetInnerValueForDebug());
        }
    }
    Result GetSystemEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = sf::NativeHandle(m_Event.GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }
    Result Cancel() NN_NOEXCEPT
    {
        Cancellable::Cancel();
        NN_RESULT_SUCCESS;
    }
    Result HasDone(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        *pOut = m_Event.TryWait();
        NN_RESULT_SUCCESS;
    }
    Result GetResult() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Event.TryWait(), detail::ResultInvalidProtocol());
        return m_Result;
    }
};

}}} // ~namespace nn::migration::detail
