﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/migration/detail/migration_Mutex.h>

namespace nn { namespace migration { namespace detail {

class AbstractStorage
{
private:
    mutable Mutex m_Lock;

protected:
    AbstractStorage() NN_NOEXCEPT
    {
        m_Lock = NN_MIGRATION_DETAIL_MUTEX_INITIALIZER(false);
    }

    virtual void CleanupImpl() const NN_NOEXCEPT = 0;

    virtual Result CreateImpl(const char* path, size_t sizeToReserve) const NN_NOEXCEPT = 0;
    virtual Result DeleteImpl(const char* path) const NN_NOEXCEPT = 0;
    virtual Result MoveImpl(const char* to, const char* from) const NN_NOEXCEPT = 0;
    virtual Result GetSizeImpl(size_t* pOutSize, const char* path) const NN_NOEXCEPT = 0;

    virtual Result CommitImpl() const NN_NOEXCEPT = 0;
    virtual Result WriteImpl(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT = 0;
    virtual Result AppendImpl(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT = 0;
    virtual Result ReadImpl(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT = 0;

public:
    LockGuard AcquireWriteLock() const NN_NOEXCEPT
    {
        return LockGuard(m_Lock);
    }

    void Cleanup() const NN_NOEXCEPT
    {
        CleanupImpl();
    }

    Result Create(const char* path, size_t sizeToReserve) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
        return CreateImpl(path, sizeToReserve);
    }
    Result Delete(const char* path) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
        return DeleteImpl(path);
    }
    Result Move(const char* to, const char* from) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
        return MoveImpl(to, from);
    }
    Result GetSize(size_t* pOutSize, const char* path) const NN_NOEXCEPT
    {
        return GetSizeImpl(pOutSize, path);
    }

    Result Commit() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
        return CommitImpl();
    }
    Result Write(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
        return WriteImpl(path, buffer, bufferSize);
    }
    Result Append(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
        return AppendImpl(path, offset, buffer, bufferSize);
    }
    Result Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT
    {
        return ReadImpl(pOutActualSize, buffer, bufferSize, path, offset);
    }
    Result Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path) const NN_NOEXCEPT
    {
        return Read(pOutActualSize, buffer, bufferSize, path, 0);
    }
};

}}} // ~namespace nn::migration::detail
