﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/manu/manu_Types.h>

namespace nn { namespace manu {

void InitializeUfio() NN_NOEXCEPT;
void FinalizeUfio() NN_NOEXCEPT;

/**
 * @brief       ホストPC に存在するファイルのファイルサイズを取得します。
 *
 * @details     ホストPC に存在する指定されたファイルのサイズを取得します。
 *
 * @param[out]   pOutData                      	取得したファイルサイズ
 * @param[in]    path                   		ファイルのパス
 */
nn::Result GetFileSize(size_t* pOutData, const char* path) NN_NOEXCEPT;

/**
 * @brief       ホストPC に存在するファイルを読み込みます。
 *
 * @details     ホストPC に存在する指定されたファイルをバッファに読み込みます。
 *
 * @param[out]   pOutBuffer                      	読み込んだデータを格納するバッファ
 * @param[in]    bufferSize                   		バッファのサイズ
 * @param[in]    path                   			ファイルのパス
 * @param[in]    offset                   			読み込むデータの開始位置
 * @param[in]    readSize                   		読み込むデータのサイズ
 */
nn::Result ReadFromHost(void* pOutBuffer, size_t bufferSize, const char* path, size_t offset, size_t readSize) NN_NOEXCEPT;

/**
 * @brief       バッファのデータをホストPC にファイルとして書き出します。
 *
 * @details     バッファのデータをホストPC にファイルとして書き出します。
 *
 * @param[in]	 pBuffer                      		書き出したいデータのバッファ
 * @param[in]    bufferSize                   		バッファのサイズ
 * @param[in]    path                   			ファイルのパス
 * @param[in]    offset                   			書き出すファイルの開始位置
 * @param[in]    writeSize                   		書き出すデータのサイズ
 */
nn::Result WriteToHost(const void* pBuffer, size_t bufferSize, const char* path, size_t offset, size_t writeSize) NN_NOEXCEPT;

/**
 * @brief       USB 経由でホスト PC からデータを受信します。
 *
 * @details     USB 経由でホスト PC からデータを受信します。
 *
 * @param[out]	 pOutSuccessReadSize          		実際に受信したデータのサイズ
 * @param[out]   pOutBuffer                   		受信したデータを格納するバッファ
 * @param[in]    bufferSize                   		バッファのサイズ
 * @param[in]    readSize                   		受信するデータのサイズ
 */
nn::Result UsbRead(size_t* pOutSuccessReadSize, void* pOutBuffer, size_t bufferSize, size_t readSize) NN_NOEXCEPT;

/**
 * @brief       USB 経由でホスト PC にデータを送信します。
 *
 * @details     USB 経由でホスト PC にデータを送信します。
 *
 * @param[out]	 pOutSuccessWriteSize          		実際に送信したデータのサイズ
 * @param[in]    pBuffer                   		送信したいデータのバッファ
 * @param[in]    bufferSize                   		バッファのサイズ
 * @param[in]    writeSize                   		送信するデータのサイズ
 */
nn::Result UsbWrite(size_t* pOutSuccessWriteSize, const void* pBuffer, size_t bufferSize, size_t writeSize) NN_NOEXCEPT;

/**
 * @brief       秘密鍵の検証をします。
 *
 * @details     較正情報内に適切な秘密鍵および証明書が書き込まれているか検証します。
 *
 * @param[out]	 outIsSuccess               		検証結果の合否
 */
nn::Result ValidatePrivateKey(bool& outIsSuccess) NN_NOEXCEPT;

/**
 * @brief       秘密鍵の検証をします。
 *
 * @details     較正情報内に適切な秘密鍵および証明書が書き込まれているか検証します。
 *
 * @param[out]	 outIsSuccess               		検証結果の合否
 * @param[in]    pLog                       		ログ用の関数ポインタ
 */
nn::Result ValidatePrivateKey(bool& outIsSuccess, const LogFunctionPointer pLog) NN_NOEXCEPT;

}}
