﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Types.h>

namespace nn { namespace ldn
{
    // 優先度の最小値です。値が低いほど優先度が高いです。
    const int PriorityMin = 0;

    // 優先度の最大値です。値が低いほど優先度が高いです。
    const int PriorityMax = 255;

    // アプリケーションの優先度です。値が低いほど優先度が高いです。
    const int UserPriority = 90;

    // システムの標準の優先度です。値が低いほど優先度が高いです。
    const int DefaultSystemPriority = 56;

   /**
    * @brief        IPv4 アドレスの割り当てを指定するエントリの最大数です。
    */
    const int AddressEntryCountMax = 8;

    /**
     * @brief       IPv4 アドレスの割り当てを指定するエントリです。
     *
     * @details
     * ノードに対して静的に IPv4 アドレスを割り当てたい場合に使用します。
     *
     * ノードに対して割り当てたい IPv4 アドレスを ipv4Address に指定し、
     * 対象となるノードを特定するために必要な MAC アドレスを macAddress に指定します。
     *
     * AddressEntry は @ref CreateNetworkPrivate 関数の引数として、
     * 最大 @ref AddressEntryCountMax 個指定することができます。
     * ここで指定した MAC アドレスをもつノードがネットワークに接続してきた場合、
     * 必ず AddressEntry で指定した IPv4 アドレスが割り当てられます。
     */
    struct AddressEntry
    {
        //! ノードに割り当てる IPv4 アドレスです。
        Ipv4Address             ipv4Address;

        //! ノードの IPv4 アドレスです。
        MacAddress              macAddress;
        Bit8                    _reserved[2];
    };
    NN_STATIC_ASSERT(sizeof(AddressEntry) == 12);

    /**
     * @brief       ネットワークの構築や接続に使用するセキュリティ・パラメータです。
     *
     * @details
     * ネットワークの構築や接続に使用するセキュリティ・パラメータです。
     *
     * @ref CreateNetwork 関数や @ref Connect 関数を使用してネットワークを構築する場合、
     * システムが内部で生成したセキュリティ・パラメータを使用するため、
     * セキュリティ・パラメータを引数として渡す必要はありません。
     * この場合、アクセスポイントが生成したセキュリティ・パラメータを、
     * ビーコンを通じてステーションと共有する必要がありますので、
     * ネットワークに接続する前にステーション側で必ず @ref Scan 関数を実行しなければなりません。
     * @ref Scan 関数の実行には数百ミリ秒以上要することから、
     * このことが速やかにネットワークを構築する上での障害となってしまいます。
     *
     * しかし、一度 @ref CreateNetwork 関数や @ref Connect 関数でネットワークを構築した後、
     * アクセスポイントやステーションといった役割の交代や、再起動後の合流が必要な場合には、
     * 前回のセキュリティ・パラメータを使いまわすことで、
     * @ref Scan 関数の実行を省略して速やかにネットワークを再構築することができます。
     *
     * そのためには、最初のネットワークを構築した後、ネットワークの参加者全員が
     * @ref GetSecurityParameter 関数を使用してセキュリティ・パラメータを取得しておきます。
     * そして、ネットワークを再構築する際にそのセキュリティ・パラメータを引数として、
     * 再構築後のネットワークでアクセスポイントになる場合は @ref CreateNetworkPrivate 関数を、
     * ステーションになる場合には @ref ConnectPrivate 関数を実行してください。
     */
    struct SecurityParameter
    {
        Bit8                    _serverRandom[RandomSize];
        SessionId               _sessionId;
    };
    NN_STATIC_ASSERT(sizeof(SecurityParameter) == 32);

}} // end of namespace nn::ldn
