﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         LDN のプロトコルバージョンです。
     */
    struct Version
    {
        uint8_t _raw;
    };

    /**
     * @brief         プロトコルバージョンを生成します。
     *
     * @param[in]     major     プロトコルのメジャーバージョンです。
     * @param[in]     minor     プロトコルのマイナーバージョンです。
     *
     * @return        プロトコルバージョンです。
     *
     * @pre
     *                - 0 <= major && major < 16
     *                - 0 <= minor && minor < 16
     */
    inline const Version MakeVersion(int major, int minor) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(major, 0, 16);
        NN_SDK_REQUIRES_RANGE(minor, 0, 16);
        Version version = {static_cast<uint8_t>((major << 4) | minor)};
        return version;
    }

    //! プロトコルのメジャーバージョンです。
    const int CurrentMajorVersion = 0;

    //! プロトコルのマイナーバージョンです。
    const int CurrentMinorVersion = 2;

    //! プロトコルバージョンです。
    const Version CurrentVersion = MakeVersion(CurrentMajorVersion, CurrentMinorVersion);

    /**
     * @brief         プロトコルバージョンからメジャーバージョンを取得します。
     *
     * @param[in]     プロトコルバージョンです。
     *
     * @return        プロトコルのメジャーバージョンです。
     */
    inline int GetMajorVersion(Version version) NN_NOEXCEPT
    {
        return version._raw >> 4;
    }

    /**
     * @brief         プロトコルバージョンからマイナーバージョンを取得します。
     *
     * @param[in]     プロトコルバージョンです。
     *
     * @return        プロトコルのマイナーバージョンです。
     */
    inline int GetMinorVersion(Version version) NN_NOEXCEPT
    {
        return version._raw & 0x0F;
    }

    /**
     * @brief         プロトコルバージョンを比較します。
     *
     * @param[in]     lhs         比較対象のバージョン
     * @param[in]     rhs         比較対象のバージョン
     *
     * @return        プロトコルバージョンが等しい場合に true です。
     */
    inline bool operator == (Version lhs, Version rhs) NN_NOEXCEPT
    {
        return lhs._raw == rhs._raw;
    }

    /**
     * @brief         プロトコルバージョンを比較します。
     *
     * @param[in]     lhs         比較対象のバージョン
     * @param[in]     rhs         比較対象のバージョン
     *
     * @return        プロトコルバージョンが等しくない場合に true です。
     */
    inline bool operator != (Version lhs, Version rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    /**
     * @brief         プロトコルバージョンを比較します。
     *
     * @param[in]     lhs         比較対象のバージョン
     * @param[in]     rhs         比較対象のバージョン
     *
     * @return        lhs の方が rhs よりもバージョンが低い場合に true です。
     */
    inline bool operator < (Version lhs, Version rhs) NN_NOEXCEPT
    {
        return lhs._raw < rhs._raw;
    }

    /**
     * @brief         プロトコルバージョンを比較します。
     *
     * @param[in]     lhs         比較対象のバージョン
     * @param[in]     rhs         比較対象のバージョン
     *
     * @return        lhs の方が rhs よりもバージョンが低いか等しい場合に true です。
     */
    inline bool operator <= (Version lhs, Version rhs) NN_NOEXCEPT
    {
        return lhs < rhs || lhs == rhs;
    }

    /**
     * @brief         プロトコルバージョンを比較します。
     *
     * @param[in]     lhs         比較対象のバージョン
     * @param[in]     rhs         比較対象のバージョン
     *
     * @return        lhs の方が rhs よりもバージョンが高い場合に true です。
     */
    inline bool operator > (Version lhs, Version rhs) NN_NOEXCEPT
    {
        return !(lhs <= rhs);
    }

    /**
     * @brief         プロトコルバージョンを比較します。
     *
     * @param[in]     lhs         比較対象のバージョン
     * @param[in]     rhs         比較対象のバージョン
     *
     * @return        lhs の方が rhs よりもバージョンが高いか等しい場合に true です。
     */
    inline bool operator >= (Version lhs, Version rhs) NN_NOEXCEPT
    {
        return !(lhs < rhs);
    }

    /**
     * @brief         指定されたバージョンと互換性があるか判定します。
     *
     * @param[in]     rhs       比較対象のバージョンです。
     *
     * @return        互換性がある場合に true です。
     *
     * @details       0.1 ～ 0.15 は互換性があるものとみなします。
     *                1.0 以上は互換性がありません。
     */
    inline bool IsCompatibleVersion(Version rhs) NN_NOEXCEPT
    {
        return (MakeVersion(0, 0) < rhs) && (rhs < MakeVersion(1, 0));
    };

}}} // namespace nn::ldn::detail
