﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/ldn_Core.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceRequest.h>
#include <nn/os.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    /**
     * @brief           LocalCommunicationServiceManager の初期化に使用するバッファです。
     */
    struct LocalCommunicationServiceManagerBuffer
    {
        char networkInterfaceMonitorThreadStack[4096];
    };

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief           ローカル通信サービスの利用が競合しないように管理します。
     */
    class LocalCommunicationServiceManager
    {
    public:

        /**
         * @brief       初期化に必要なバッファサイズです。
         */
        static const size_t RequiredBufferSize =
            sizeof(impl::LocalCommunicationServiceManagerBuffer);

        /**
         * @brief       コンストラクタです。
         * @param[in]   buffer                  初期化に使用するバッファです。
         * @param[in]   bufferSize              初期化に使用するバッファのサイズです。
         * @param[in]   core                    ローカル通信のコア機能を提供するオブジェクトです。
         */
        LocalCommunicationServiceManager(
            void* buffer, size_t bufferSize, Core* pCore) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~LocalCommunicationServiceManager() NN_NOEXCEPT;

        /**
         * @brief       ローカル通信機能の利用に必要なコンテキストを生成します。
         * @return      生成されたコンテキストです。
         */
        int CreateContext() NN_NOEXCEPT;

        /**
         * @brief       コンテキストを破棄します。
         * @param[in]   contextId               コンテキストです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result DestroyContext(int contextId) NN_NOEXCEPT;

        /**
         * @brief       ローカル通信機能の利用を開始します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   pid                     クライアントプロセスの識別子です。
         * @param[in]   param                   ネットワーク・インタフェースに対する要求です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result StartLocalCommunication(
            int contextId, Bit64 pid, const NetworkInterfaceRequestParameter& param) NN_NOEXCEPT;

        /**
         * @brief       ローカル通信機能の使用を終了します。
         * @param[in]   contextId               コンテキストです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result StopLocalCommunication(int contextId) NN_NOEXCEPT;

        /**
         * @brief       現在の状態を取得します。
         * @param[in]   contextId               コンテキストです。
         * @return      現在の状態です。
         */
        State GetState(int contextId) NN_NOEXCEPT;

        /**
         * @brief       周囲のネットワークを探索します。
         * @param[in]   contextId               コンテキストです。
         * @param[out]  pOutScanResultArray     スキャン結果の出力先です。
         * @param[out]  pOutCount               pOutScanResultArray に出力されたスキャン結果の数です。
         * @param[in]   bufferCount             pOutScanResultArray に格納できるスキャン結果の数です。
         * @param[in]   filter                  スキャンの条件です
         * @param[in]   channel                 スキャン対象のチャンネルです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result Scan(
            int contextId,
            NetworkInfo* pOutScanResultArray,
            int* pOutCount,
            int bufferCount,
            const ScanFilter& filter,
            int channel) NN_NOEXCEPT;

        /**
         * @brief       現在参加しているネットワークの情報を取得します。
         * @param[in]   contextId               コンテキストです。
         * @param[out]  pOutNetwork             ネットワーク情報の出力先です。
         * @param[out]  pOutUpdates             ネットワークに参加しているノードの変化の出力先です。
         * @param[in]   bufferCount             pOutUpdates に格納できる NodeLatestUpdate の数です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result GetNetworkInfo(
            int contextId, NetworkInfo* pOutNetwork,
            NodeLatestUpdate* outUpdates, int bufferCount) NN_NOEXCEPT;

        /**
         * @brief       現在参加しているネットワークの情報を取得します。
         * @param[in]   contextId               コンテキストです。
         * @param[out]  pOutNetwork             ネットワーク情報の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result GetNetworkInfo(
            int contextId, NetworkInfo* pOutNetwork) const NN_NOEXCEPT;

        /**
         * @brief       自身の IP アドレスを取得します。
         * @param[out]  pOutAddress             IP アドレスの出力先です。
         * @param[out]  pOutMask                サブネットマスクの出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result GetIpv4Address(
            int contextId, Ipv4Address* pOutAddress, SubnetMask* pOutMask) const NN_NOEXCEPT;

        /**
         * @brief       現在参加しているネットワークのセキュリティ・パラメータを取得します。
         * @param[in]   contextId               コンテキストです。
         * @param[out]  pOutParam               セキュリティ・パラメータの出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result GetSecurityParameter(
            int contextId, SecurityParameter* pOutParam) const NN_NOEXCEPT;

       /**
        * @brief        現在参加しているネットワークの設定情報を取得します。
         * @param[in]   contextId               コンテキストです。
        * @param[out]   pOutNetworkConfig       ネットワークの設定情報の出力先です。
        * @retresult
        *  @handleresult{ResultSuccess}
        *  @handleresult{ResultInvalidState}
        *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
        * @endretresult
        */
        Result GetNetworkConfig(
            int contextId, NetworkConfig* pOutNetworkConfig) const NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントモードで通信を開始します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   pStateChangeEvent       状態遷移時にシグナルするイベントです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result OpenAccessPoint(
            int contextId, nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントモードを終了します。
         * @param[in]   contextId               コンテキストです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result CloseAccessPoint(int contextId) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントとして新規にネットワークを構築します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   network                 構築するネットワークの設定情報です。
         * @param[in]   securityConfig          構築するネットワークのセキュリティ設定です。
         * @param[in]   securityParam           構築するネットワークのセキュリティ・パラメータです。
         * @param[in]   user                    ネットワークを構築するユーザの情報です。
         * @param[in]   addressEntryCount       静的に割り当てる IPv4 アドレスの数です。
         * @param[in]   addressEntries          静的に割り当てる IPv4 アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result CreateNetwork(
            int contextId,
            const NetworkConfig& network,
            const SecurityConfig& securityConfig,
            const SecurityParameter& securityParam,
            const UserConfig& user,
            int addressEntryCount,
            const AddressEntry* addressEntries) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントとして構築したネットワークを破棄します。
         * @param[in]   contextId               コンテキストです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result DestroyNetwork(int contextId) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントが配信する任意データを設定します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   data                    配信するデータです。
         * @param[in]   dataSize                配信するデータのサイズです。
         *                                      @ref AdvertiseDataSizeMax 以下の値を設定します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result SetAdvertiseData(int contextId, const void* data, size_t dataSize) NN_NOEXCEPT;

        /**
         * @brief       アクセスポイントにおけるステーションの接続受付ポリシーを変更します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   policy                  新しい接続受付ポリシーです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result SetStationAcceptPolicy(int contextId, AcceptPolicy policy) NN_NOEXCEPT;

        /**
         * @brief         指定したステーションをネットワークから追放します。
         * @param[in]     contextId             コンテキストです。
         * @param[in]     ipv4Address           対象のステーションの IPv4 アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultNodeNotFound}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         *  @handleresult{ResultDeviceOccupied}
         * @endretresult
         */
        Result Reject(int contextId, Ipv4Address ipv4Address) NN_NOEXCEPT;

        /**
         * @brief         ステーションの接続を制限するフィルタにエントリを追加します。
         * @param[in]     contextId             コンテキストです。
         * @param[in]     macAddress            対象のステーションの MAC アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result AddAcceptFilterEntry(int contextId, MacAddress macAddress) NN_NOEXCEPT;

        /**
         * @brief         フィルタに登録されているエントリを全て削除します。
         * @param[in]     contextId             コンテキストです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result ClearAcceptFilter(int contextId) NN_NOEXCEPT;

        /**
         * @brief       ステーションモードで通信を開始します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   pStateChangeEvent       状態遷移時にシグナルするイベントです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result OpenStation(
            int contextId, nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT;

        /**
         * @brief       ステーションモードを終了します。
         * @param[in]   contextId               コンテキストです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result CloseStation(int contextId) NN_NOEXCEPT;

        /**
         * @brief       ステーションとして指定されたネットワークに接続します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   network                 接続先のネットワークの情報です。
         * @param[in]   securityConfig          接続に必要なセキュリティパラメータです。
         * @param[in]   securityParam           接続先のネットワークのセキュリティ・パラメータです。
         * @param[in]   user                    ネットワークに参加するユーザの情報です。
         * @param[in]   version                 ローカル通信バージョンです。
         * @param[in]   option                  オプションです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultNetworkNotFound}
         *  @handleresult{ResultConnectionTimeout}
         *  @handleresult{ResultConnectionRejected}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result Connect(
            int contextId,
            const NetworkConfig& network,
            const SecurityConfig& securityConfig,
            const SecurityParameter& securityParam,
            const UserConfig& user,
            int version,
            ConnectOption option) NN_NOEXCEPT;

        /**
         * @brief       ステーションとして接続中のネットワークから切断します。
         * @param[in]   contextId               コンテキストです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result Disconnect(int contextId) NN_NOEXCEPT;

        /**
         * @brief       切断理由を取得します。
         * @param[in]   contextId               コンテキストです。
         * @param[out:  pOutReason              切断理由の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         */
        Result GetDisconnectReason(int contextId, DisconnectReason* pOutReason) const NN_NOEXCEPT;

        /**
         * @brief       ローカル通信の動作モードを設定します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   mode                    ローカル通信の動作モードです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         * @endretresult
         */
        Result SetOperationMode(int contextId, OperationMode mode) NN_NOEXCEPT;

        /**
         * @brief       無線コントローラーの接続制限を有効化あるいは無効化します。
         * @param[in]   contextId               コンテキストです。
         * @param[in]   restriction             接続制限の設定です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultBadRequest}
         *  @handleresult{ResultWifiOff}
         *  @handleresult{ResultSleep}
         *  @handleresult{ResultDeviceOccupied}
         * @endretresult
         */
        Result SetWirelessControllerRestriction(
            int contextId, WirelessControllerRestriction restriction) NN_NOEXCEPT;

    private:

        Result IsActiveContext(int contextId) const NN_NOEXCEPT;

        Result StopLocalCommunicationImpl(int contextId, bool isTriggeredBySystem) NN_NOEXCEPT;

        static void NetworkInterfaceMonitorThread(void* pArg) NN_NOEXCEPT;

        mutable nn::os::Mutex m_Mutex;
        nn::os::Event m_StartLocalCommunicationEvent;
        nn::os::Event m_StopLocalCommunicationEvent;
        nn::os::Event m_ReleaseNetworkInterfaceEvent;
        nn::os::Event m_CancelEvent;
        nn::os::ThreadType m_NetworkInterfaceMonitorThread;
        NetworkInterfaceRequest m_ActiveRequest;
        impl::LocalCommunicationServiceManagerBuffer* m_Buffer;
        Core* m_pCore;
        int m_ActiveContextId;
    };

}}} // namespace nn::ldn::detail
