﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    Bit32 RandomEngine() NN_NOEXCEPT;

    template <size_t N>
    struct BitTypeRandomGenerator;

    template<>
    struct BitTypeRandomGenerator<1>
    {
        typedef Bit8 BitType;
        static BitType Generate() NN_NOEXCEPT
        {
            return static_cast<BitType>(RandomEngine() & UINT8_C(0xFF));
        }
    };

    template<>
    struct BitTypeRandomGenerator<2>
    {
        typedef Bit16 BitType;
        static BitType Generate() NN_NOEXCEPT
        {
            return static_cast<BitType>(RandomEngine() & UINT16_C(0xFFFF));
        }
    };

    template<>
    struct BitTypeRandomGenerator<4>
    {
        typedef Bit32 BitType;
        static BitType Generate() NN_NOEXCEPT
        {
            return RandomEngine();
        }
    };

    template<>
    struct BitTypeRandomGenerator<8>
    {
        typedef Bit64 BitType;
        static BitType Generate() NN_NOEXCEPT
        {
            return (static_cast<BitType>(RandomEngine()) << 32) | RandomEngine();
        }
    };

    template <typename T, typename Enable = void>
    struct IntegralTypeRandomGenerator;

    template <typename T>
    struct IntegralTypeRandomGenerator<
        T, typename std::enable_if<std::is_integral<T>::value>::type>
    {
        static T Generate() NN_NOEXCEPT
        {
            return static_cast<T>(BitTypeRandomGenerator<sizeof(T)>::Generate());
        }
    };

    template <typename T>
    inline T RandomDispatcher(std::true_type) NN_NOEXCEPT
    {
        return IntegralTypeRandomGenerator<T>::Generate();
    }

    template <typename T>
    inline T RandomDispatcher(std::false_type) NN_NOEXCEPT
    {
        T output;
        RandomImpl(&output);
        return output;
    }

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         疑似乱数を生成します。
     *
     * @tparam        T             生成する疑似乱数の型です。
     *
     * @return        生成された疑似乱数です。
     */
    template <typename T>
    inline T Random() NN_NOEXCEPT
    {
        return impl::RandomDispatcher<T>(std::is_integral<T>());
    }

    /**
     * @brief         疑似乱数を生成します。
     *
     * @param[out]    out           疑似乱数の出力先です。
     * @param[in]     size          出力サイズです。
     *
     * @return        生成された疑似乱数です。
     */
    void FillRandom(void* out, size_t size) NN_NOEXCEPT;

}}} // end of namespace nn::ldn::detail
