﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceBase.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    struct WlanActionFrameMonitorBuffer
    {
        //! 受信スレッド用のバッファです。
        char monitorThreadStack[4096];

        //! 受信用のバッファです。
        char receive[2048];
    };

    struct WlanBuffer
    {
        //! Action Frame の送受信用のバッファです。
        WlanActionFrameMonitorBuffer actionFrameMonitor;

        //! フレーム送信用のバッファです。
        char send[2048];

        //! スキャン用のバッファです。
        union
        {
            char wlan[512 * 24];
            L2ScanResult result;
        } scan;
    };

    class WlanActionFrameMonitor
    {
        NN_DISALLOW_COPY(WlanActionFrameMonitor);
        NN_DISALLOW_MOVE(WlanActionFrameMonitor);

    public:

        explicit WlanActionFrameMonitor(WlanActionFrameMonitorBuffer* buffer) NN_NOEXCEPT;

        ~WlanActionFrameMonitor() NN_NOEXCEPT;

        void StartMonitoring(L2ScanResult* scanBuffer, int bufferCount) NN_NOEXCEPT;

        void StartMonitoring(
            L2ScanResult* scanBuffer, int bufferCount, MacAddress bssid) NN_NOEXCEPT;

        void StopMonitoring() NN_NOEXCEPT;

        nn::os::SystemEvent& GetReceivedEvent() NN_NOEXCEPT;

        int GetScanResultCount() const NN_NOEXCEPT;

        bool GetScanResult(L2ScanResult* pOutResult, int index) const NN_NOEXCEPT;

        bool GetScanResult(L2ScanResult* pOutResult, MacAddress bssid) const NN_NOEXCEPT;

        void Merge(const void* wlanScanbuffer) NN_NOEXCEPT;

        void SortByRssi() NN_NOEXCEPT;

    private:

        static void MonitorThread(void* pArg) NN_NOEXCEPT;

        nn::os::ThreadType m_MonitorThread;
        nn::os::SystemEvent m_ReceivedEvent;
        mutable nn::os::Mutex m_Mutex;
        WlanActionFrameMonitorBuffer* m_Buffer;
        L2ScanResult* m_ScanBuffer;
        int m_ScanBufferCount;
        int m_ScanResultCount;
        uint32_t m_RxEntry;
        MacAddress m_Bssid;
        bool m_IsMonitoring;
    };

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         L2 上の端末のインタフェースです。
     */
    class Wlan : public NetworkInterfaceBase
    {
    public:

        static size_t GetRequiredBufferSize() NN_NOEXCEPT;

        Wlan(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        virtual ~Wlan() NN_NOEXCEPT;

        virtual nn::os::SystemEvent& GetStateChangeEvent() NN_NOEXCEPT NN_OVERRIDE;

        virtual nn::os::SystemEvent& GetBeaconReceivedEvent() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result GetNetworkProfile(
            NetworkProfile* pOutProfile) const NN_NOEXCEPT NN_OVERRIDE;

        virtual void GetNetworkInterfaceProfile(
            NetworkInterfaceProfile* pOutProfile) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Scan(
            L2ScanResult* outBuffer, int* pOutCount, int bufferCount,
            int channel) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result OpenAccessPoint() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result SetBeaconData(const void* data, size_t dataSize) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result CreateNetwork(
            const Ssid& ssid, int channel, int nodeCountMax,
            const void* key, size_t keySize) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Reject(MacAddress address) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result GetStations(
            L2StationInfo* outStations,
            int* pOutCount, int bufferCount) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result DestroyNetwork() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result CloseAccessPoint() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result OpenStation() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Connect(
            const Ssid& ssid, int channel, int nodeCountMax,
            const void* key, size_t keySize) NN_NOEXCEPT NN_OVERRIDE;

        virtual bool GetBeaconData(L2ScanResult* pOutBeacon) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Disconnect() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result GetConnectionStatus(L2StationInfo* pOut) const NN_NOEXCEPT NN_OVERRIDE;

        virtual Result CloseStation() NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Send(const void* data, size_t dataSize) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Receive(
            void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result Cancel() NN_NOEXCEPT NN_OVERRIDE;

        virtual L2State GetState() const NN_NOEXCEPT NN_OVERRIDE;

        virtual void SetOperationMode(OperationMode mode) NN_NOEXCEPT NN_OVERRIDE;

        virtual void SetWirelessControllerRestriction(
            WirelessControllerRestriction restriction) NN_NOEXCEPT NN_OVERRIDE;

    private:

        virtual int ConvertRssiToLinkLevel(int rssi) const NN_NOEXCEPT NN_OVERRIDE;

        Result SetWlanMode(int nodeCountMax) NN_NOEXCEPT;

        uint32_t CreateRxEntry() const NN_NOEXCEPT;

        void DeleteRxEntry(uint32_t rxIndex) NN_NOEXCEPT;

        Result ConnectWithRetry(
            MacAddress*pOutBssid, const Ssid& ssid, int channel, const void* key,
            size_t keySize, int retryCountMax) NN_NOEXCEPT;

        impl::WlanBuffer* m_WlanBuffer;
        impl::WlanActionFrameMonitor m_Monitor;
        nn::os::SystemEvent m_StateChangeEvent;
        nn::os::SystemEvent m_BtmEvent;
        L2State m_Status;
        MacAddress m_MacAddress;
        MacAddress m_Bssid;
        OperationMode m_OperationMode;
        WirelessControllerRestriction m_WirelessControllerRestriction;
        uint32_t m_RxIndex;
        bool m_IsDistributing;
    };

}}} // namespace nn::ldn::detail
