﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterface.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceRequestParameter.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    /**
     * @brief       ネットワークインタフェースの占有をリクエストするクラスです。
     */
    class NetworkInterfaceRequestImpl
    {
        NN_DISALLOW_COPY(NetworkInterfaceRequestImpl);

    public:

        /**
         * @brief       コンストラクタです。
         */
        NetworkInterfaceRequestImpl() NN_NOEXCEPT;

        /**
         * @brief       ムーブコンストラクタです。
         * @param[in]   other       移動元のオブジェクトです。
         */
        NetworkInterfaceRequestImpl(NetworkInterfaceRequestImpl&& other) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~NetworkInterfaceRequestImpl() NN_NOEXCEPT;

        /**
         * @brief       移動代入演算子です。
         * @param[in]   other       移動元のオブジェクトです。
         */
        NetworkInterfaceRequestImpl& operator = (NetworkInterfaceRequestImpl&& other) NN_NOEXCEPT;

        /**
         * @brief       ネットワークインタフェースの占有を要求します。
         * @param[in]   param       ネットワークインタフェースに対する要求です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultDeviceDisabled}
         * @endretresult
         */
        Result Submit(const NetworkInterfaceRequestParameter& param) NN_NOEXCEPT;

        /**
         * @brief       自身が占有しているネットワークインタフェースを解放します。
         */
        void Release() NN_NOEXCEPT;

        /**
         * @brief       占有中のネットワークインタフェースを取得します。
         * @return      占有中のネットワークインタフェースです。
         */
        INetworkInterface* GetNetworkInterface() NN_NOEXCEPT;

        /**
         * @brief       ネットワークインタフェースの状態を取得します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultDeviceDisabled}
         * @endretresult
         */
        Result GetResult() const NN_NOEXCEPT;

        /**
         * @brief       ネットワークインタフェースの占有状態変更を通知するイベントを取得します。
         * @return      ネットワークインタフェースの占有状態変更を通知するイベントです。
         */
        nn::os::SystemEvent& GetRequestStateChangeEvent() NN_NOEXCEPT;

    private:

        nn::os::SystemEvent m_SystemEvent;
        INetworkInterface* m_pNetworkInterface;
        Result m_LastError;
        NetworkInterfaceRequestParameter m_Param;
    };

}}}} // namespace nn::ldn::detail::impl
