﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn/ldn_MacAddress.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkProfile.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceProfile.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace ldn { namespace detail
{
    //! RSSI の最小値です。
    const int RssiMin = -128;

    /**
     * @brief         L2 の状態です。
     */
    enum L2State
    {
        L2State_None,                   //!< インタフェースは未使用です。
        L2State_AccessPoint,            //!< アクセスポイントとしての動作を開始しました。
        L2State_AccessPointCreated,     //!< アクセスポイントとしてネットワークを構築しています。
        L2State_Station,                //!< ステーションとしての動作を開始しました。
        L2State_StationConnected,       //!< ステーションとしてネットワークに接続しています。
    };

    /**
     * @brief         ステーションの接続状態です。
     */
    enum L2StationState
    {
        //! ステーションは切断されています。
        L2StationState_Disconnected,

        //! ステーションは接続中です。
        L2StationState_Connected
    };

    /**
     * @brief         ステーションの切断理由です。
     */
    enum L2DisconnectReason
    {
        //! 原因不明です。
        L2DisconnectReason_Unknown,

        //! ビーコンロストです。
        L2DisconnectReason_BeaconLost,

        //! ステーションの要求による切断です。
        L2DisconnectReason_DisconnectCommand,

        //! アクセスポイントの要求による切断です。
        L2DisconnectReason_DisconnectedByAccessPoint,
    };

    /**
     * @brief         ステーションの接続状態です。
     */
    struct L2StationInfo
    {
        //! ステーションの AID です。
        int8_t aid;

        //! ステーションの状態です。
        Bit8 status;

        //! ステーションの MAC アドレスです。
        MacAddress macAddress;

        //! 接続時刻です。
        int64_t connectedAt;

        //! 切断理由です。
        int8_t disconnectReason;

        //! 通信品質です。
        int8_t rssi;
        NN_PADDING6;
    };

    /**
     * @brief         スキャン結果を格納する構造体です。
     */
    struct L2ScanResult
    {
        //  アクセスポイントの BSSID です。
        MacAddress          bssid;

        //! アクセスポイントの SSID です。
        Ssid                ssid;

        //! 無線チャンネルです。
        int16_t             channel;

        //! RSSI です。
        int8_t              rssi;
        NN_PADDING7;

        //! 受信したスキャンデータのサイズです。
        uint16_t dataSize;

        //! スキャンデータの本体です。
        Bit8 data[1484];
    };
    NN_STATIC_ASSERT(sizeof(L2ScanResult) == 1536);

    /**
     * @brief         L2 上の端末のインタフェースです。
     */
    class INetworkInterface
    {
    public:

        /**
         * @brief       スコープ内限定でネットワークに接続します。
         */
        class ScopedConnection
        {
        public:

            ScopedConnection(
                INetworkInterface* pNetworkInterface, const Ssid& ssid,
                int channel, int nodeCountMax, const void* key, size_t keySize) NN_NOEXCEPT
                : m_pNetworkInterface(pNetworkInterface)
            {
                m_Result = m_pNetworkInterface->Connect(
                    ssid, channel, nodeCountMax , key, keySize);
            }

            ~ScopedConnection() NN_NOEXCEPT
            {
                if (m_Result.IsSuccess() && m_pNetworkInterface != nullptr)
                {
                    m_pNetworkInterface->Disconnect();
                }
            }

            Result GetResult() const NN_NOEXCEPT
            {
                return m_Result;
            }

            void Detach() NN_NOEXCEPT
            {
                m_pNetworkInterface = nullptr;
            }

        private:

            INetworkInterface* m_pNetworkInterface;
            Result m_Result;
        };


        /**
         * @brief       デストラクタです。
         */
        virtual ~INetworkInterface() NN_NOEXCEPT
        {
        }

        /**
         * @brief       接続状態の変化を通知するイベントを取得します。
         * @return      接続状態の変化を通知するイベントです。
         */
        virtual nn::os::SystemEvent& GetStateChangeEvent() NN_NOEXCEPT = 0;

        /**
         * @brief       ビーコン受信を通知するイベントを取得します。
         * @return      ビーコン受信を通知するイベントです。
         */
        virtual nn::os::SystemEvent& GetBeaconReceivedEvent() NN_NOEXCEPT = 0;

        /**
         * @brief       接続中のネットワークの詳細を取得します。
         * @param[out]  pOutProfile ネットワーク情報の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result GetNetworkProfile(NetworkProfile* pOutProfile) const NN_NOEXCEPT = 0;

        /**
         * @brief       ネットワークインタフェースの詳細を取得します。
         * @param[out]  pOutProfile ネットワークインタフェース情報の出力先です。
         */
        virtual void GetNetworkInterfaceProfile(
            NetworkInterfaceProfile* pOutProfile) const NN_NOEXCEPT = 0;

        /**
         * @brief       周囲のネットワークを探索します。
         * @param[out]  outBuffer       スキャン結果の出力先です。
         * @param[out]  pOutSize        見つかったネットワークの数です。
         * @param[in]   bufferCount     outBuffer のバッファサイズです。
         * @param[in]   channel         スキャン対象の無線チャンネルです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result Scan(
            L2ScanResult* outBuffer, int* pOutSize, int bufferCount, int channel) NN_NOEXCEPT = 0;

        /**
         * @brief       アクセスポイントとしての動作を開始します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result OpenAccessPoint() NN_NOEXCEPT = 0;

        /**
         * @brief       ビーコンで配信するデータを設定します。
         * @param[in]   data            配信するデータです。
         * @param[in]   dataSize        配信するデータのサイズです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result SetBeaconData(const void* data, size_t dataSize) NN_NOEXCEPT = 0;

        /**
         * @brief       ネットワークを構築します。
         * @param[in]   ssid            ネットワークの SSID です。
         * @param[in]   channel         無線チャンネルです。
         * @param[in]   nodeCountMax    ノードの最大接続台数です。
         * @param[in]   key             通信の暗号化に使用する鍵です。
         * @param[in]   keySize         通信の暗号化に使用する鍵のバイトサイズです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result CreateNetwork(
            const Ssid& ssid, int channel, int nodeCountMax,
            const void* key, size_t keySize) NN_NOEXCEPT = 0;

        /**
         * @brief       ステーションを切断します。
         * @param[in]   address         対象となるステーションの MAC アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultNodeNotFound}
         * @endretresult
         */
        virtual Result Reject(MacAddress address) NN_NOEXCEPT = 0;

        /**
         * @brief       このアクセスポイントに接続中のステーションの一覧を取得します。
         * @param[out]  outStations      ステーションの出力先です。
         * @param[out]  pOutCount       接続中のステーションの総数です。自身は含みません。
         * @param[in]   bufferCount     outStations のバッファサイズです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result GetStations(
            L2StationInfo* outStations, int* pOutCount, int bufferCount) const NN_NOEXCEPT = 0;

        /**
         * @brief       ネットワークを破棄します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result DestroyNetwork() NN_NOEXCEPT = 0;

        /**
         * @brief       アクセスポイントとしての動作を終了します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result CloseAccessPoint() NN_NOEXCEPT = 0;

        /**
         * @brief       ステーションとしての動作を開始します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result OpenStation() NN_NOEXCEPT = 0;

        /**
         * @brief       ネットワークに接続します。
         * @param[in]   ssid            接続先の SSID です。
         * @param[in]   channel         無線チャンネルです。
         * @param[in]   nodeCountMax    ノードの最大接続台数です。
         * @param[in]   key             通信の暗号化に使用する鍵です。
         * @param[in]   keySize         通信の暗号化に使用する鍵のバイトサイズです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultNetworkNotFound}
         * @endretresult
         */
        virtual Result Connect(
            const Ssid& ssid, int channel, int nodeCountMax,
            const void* key, size_t keySize) NN_NOEXCEPT = 0;

        /**
         * @brief       最新のビーコンデータを取得します。
         * @param[out]  pOutBeacon      最新のビーコンデータの出力先です。
         * @return      ビーコンデータの取得に成功した場合は true です。
         */
        virtual bool GetBeaconData(L2ScanResult* pOutBeacon) const NN_NOEXCEPT = 0;

        /**
         * @brief       ネットワークから切断します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result Disconnect() NN_NOEXCEPT = 0;

        /**
         * @brief       自身の接続状態を取得します。
         * @param[out}  pOut            接続状態の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result GetConnectionStatus(L2StationInfo* pOut) const NN_NOEXCEPT = 0;

        /**
         * @brief       ステーションとしての動作を終了します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result CloseStation() NN_NOEXCEPT = 0;

        /**
         * @brief       データフレームを送信します。
         * @param[in]   data            送信するデータです。
         * @param[in]   dataSize        送信するデータのサイズです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result Send(const void* data, size_t dataSize) NN_NOEXCEPT = 0;

        /**
         * @brief       データフレームを受信します。
         * @param[in]   buffer          受信したデータを保存するバッファです。
         * @param[out]  pOutSize        受信サイズです。
         * @param[in]   bufferSize      受信バッファのサイズです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result Receive(
            void* buffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT = 0;

        /**
         * @brief       データフレームの受信をキャンセルします。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        virtual Result Cancel() NN_NOEXCEPT = 0;

        /**
         * @brief       ネットワークインタフェースの状態を取得します。
         * @return      ネットワークインタフェースの状態です。
         */
        virtual L2State GetState() const NN_NOEXCEPT = 0;

        /**
         * @brief       指定された無線チャンネルのサポート状況を確認します。
         * @param[in]   channel     対象の無線チャンネルです。
         * @return      指定された無線チャンネルをサポートしていれば true です。
         */
        virtual bool IsSupportedChannel(int channel) const NN_NOEXCEPT = 0;

        /**
         * @brief       指定された無線チャンネルが自動チャンネル選択で選択されうるかをを確認します。
         * @param[in]   channel     対象の無線チャンネルです。
         * @return      指定された無線チャンネルが自動選択される可能性があれば true です。
         */
        virtual bool IsAutoSelectableChannel(int channel) const NN_NOEXCEPT = 0;

        /**
         * @brief       通信品質を取得します。
         * @return      通信品質です。
         */
        virtual int GetLinkLevel() const NN_NOEXCEPT = 0;

        /**
         * @brief       RSSI を LinkLevel に変換します。
         * @param[in]   rssi        変換対象の RSSI です。
         * @return      RSSI を LinkLevel に変換した値です。
         */
        virtual int ConvertRssiToLinkLevel(int rssi) const NN_NOEXCEPT = 0;

        /**
         * @brief       ローカル通信の動作モードを変更します。
         * @param[in]   mode        動作モードです。
         */
        virtual void SetOperationMode(OperationMode mode) NN_NOEXCEPT = 0;

        /**
         * @brief       無線コントローラーの接続制限を設定します。
         * @param[in]   restriction 接続制限の設定です。
         */
        virtual void SetWirelessControllerRestriction(
            WirelessControllerRestriction restriction) NN_NOEXCEPT = 0;
    };

}}} // namespace nn::ldn::detail
