﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameQueue.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterface.h>
#include <nn/ldn/detail/NetworkInterface/ldn_ProtocolId.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ThreadTypes.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief       フレームの送信インタフェースです。
     */
    class IFrameSender
    {
    public:

        /**
         * @brief       デストラクタです。
         */
        virtual ~IFrameSender()
        {
        }

        /**
         * @brief       データフレームを送信キューに登録します。
         * @param[in]   data            対象のデータです。
         * @param[in]   dataSize        データサイズです。
         * @param[in]   dst             送信先の MAC アドレスです。
         * @param[in]   protocol        プロトコルです。
         * @result      処理の結果です。
         */
        virtual Result Send(
            const void* data, size_t dataSize,
            MacAddress dst, ProtocolId protocol) NN_NOEXCEPT = 0;
    };

    /**
     * @brief       フレームの送信を担当するクラスです。
     */
    class FrameSender : public IFrameSender
    {
    public:

        /**
         * @brief       スコープ内限定で送信を有効化します。
         */
        class ScopedInitializer
        {
        public:

            ScopedInitializer(
                FrameSender* pSender, INetworkInterface* pInterface, void* buffer,
                size_t bufferSize, size_t frameSizeMax, int capacity) NN_NOEXCEPT
                : m_pSender(pSender)
            {
                m_pSender->Initialize(
                    pInterface, buffer, bufferSize, frameSizeMax, capacity);
            };

            ~ScopedInitializer() NN_NOEXCEPT
            {
                if (m_pSender != nullptr)
                {
                    m_pSender->Finalize();
                }
            }

            void Detach() NN_NOEXCEPT
            {
                m_pSender = nullptr;
            }

        private:

            FrameSender* m_pSender;
        };

        /**
         * @brief       コンストラクタです。
         */
        FrameSender() NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~FrameSender() NN_NOEXCEPT;

        /**
         * @brief       初期化に必要なバッファサイズを計算します。
         * @param[in]   frameSizeMax    フレームの最大サイズです。
         * @param[in]   capacity        送信キューのサイズです。
         * @return      バッファサイズです。
         */
        static size_t GetRequiredBufferSize(size_t frameSizeMax, int capacity) NN_NOEXCEPT;

        /**
         * @brief       初期化します。
         * @param[in]   pInterface      ネットワークインタフェースです。
         * @param[in]   buffer          初期化に使用するバッファです。
         * @param[in]   bufferSize      初期化に使用するバッファのバイトサイズです。
         * @param[in]   frameSizeMax    フレームの最大サイズです。
         * @param[in]   capacity        送信キューのサイズです。
         */
        void Initialize(
            INetworkInterface* pInterface, void* buffer, size_t bufferSize,
            size_t frameSizeMax, int capacity) NN_NOEXCEPT;

        /**
         * @brief       終了します。
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief       データフレームを送信キューに登録します。
         * @param[in]   data            対象のデータです。
         * @param[in]   dataSize        データサイズです。
         * @param[in]   dst             送信先の MAC アドレスです。
         * @param[in]   protocol        プロトコルです。
         * @result      処理の結果です。
         */
        virtual Result Send(
            const void* data, size_t dataSize,
            MacAddress dst, ProtocolId protocol) NN_NOEXCEPT NN_OVERRIDE;

    private:

        static void SendThread(void* pArg) NN_NOEXCEPT;

        FrameQueue m_Queue;
        INetworkInterface* m_pNetworkInterface;
        Bit8* m_SendBuffer;
        size_t m_SendBufferSize;
        bool m_IsRunning;
        bool m_IsInitialized;
        nn::os::Event m_EnqueueEvent;
        nn::os::Event m_DequeueEvent;
        nn::os::Mutex m_Mutex;
        nn::os::ThreadType m_SendThread;
    };

}}} // end of namespace nn::ldn::detail
