﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/Authentication/ldn_Authentication.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameReceiver.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameSender.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief           認証結果の履歴です。
     */
    struct AuthenticationHistroy
    {
        //! 対象のクライアントの識別子です。
        int32_t id;

        //! 認証の結果です。
        Bit8 authenticationResult;

        //! 受信データのサイズです。
        uint8_t receivedSize;
        char _reserved[2];

        //! クライアントが生成した乱数です。
        Bit8 clientRandom[RandomSize];

        //! 受信データです。
        char receivedData[232];
    };
    NN_STATIC_ASSERT(sizeof(AuthenticationHistroy) == 256);

    /**
     * @brief           認証サーバのインタフェースです。
     */
    class IAuthenticationServer
    {
    public:

        /**
         * @brief       デストラクタです。
         */
        virtual ~IAuthenticationServer() NN_NOEXCEPT
        {
        }

        /**
         * @brief       認証サーバとしての動作を開始します。
         * @param[in]   networkId       ネットワーク識別子です。
         * @param[in]   serverRandom    サーバが生成した乱数です。
         * @param[in]   pReceiver       データ受信に使用するインタフェースです。
         * @param[in]   pSender         データ送信に使用するインタフェースです
         */
        virtual void BeginServer(
            const NetworkId& networkId, const Bit8 (&serverRandom)[RandomSize],
            IFrameReceiver* pReceiver, IFrameSender* pSender) NN_NOEXCEPT = 0;

        /**
         * @brief       認証サーバを終了します。
         */
        virtual void EndServer() NN_NOEXCEPT = 0;

        /**
         * @brief       認証を許可あるいは拒否するクライアントを追加します。
         * @param[in]   macAddress      MAC アドレスです。
         * @param[in]   accpet          許可するか拒否するかのフラグです。
         * @return      クライアントの操作に使用する識別子です。
         */
        virtual int Register(MacAddress macAddress, bool accept) NN_NOEXCEPT = 0;

        /**
         * @brief       認証許可を取り消します。
         * @param[in]   handle          Register() で取得した識別子です。
         */
        virtual void Unregister(int id) NN_NOEXCEPT = 0;

        /**
         * @brief       認証成功時にクライアントに送信するデータを設定します。
         * @param[in]   data            認証成功時にクライアントに送信するデータです。
         * @param[in]   dataSize        data のサイズです。
         */
        virtual void SetData(const void* data, size_t dataSize) NN_NOEXCEPT = 0;

        /**
         * @brief       認証が発生した場合に通知されるイベントを取得します。
         * @return      認証を発生した場合に通知されるイベントです
         */
        virtual nn::os::Event& GetAuthenticationEvent() NN_NOEXCEPT = 0;

        /**
         * @brief       最近の認証結果を取得します。
         * @param[out]  pOutHistroy     最新の認証結果です。
         * @retval      true            認証結果の取得に成功しました。
         * @retval      false           取得できる認証結果がありません。
         */
        virtual bool GetAuthenticationHistory(
            AuthenticationHistroy* pOutHistroy) NN_NOEXCEPT = 0;
    };

}}} // namespace nn::ldn::detail
