﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/Advertise/ldn_AdvertiseBuilder.h>
#include <nn/ldn/detail/Advertise/ldn_AdvertiseParser.h>
#include <nn/ldn/detail/Advertise/ldn_PlainAdvertiseImpl.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief       平文の Advertise を構築するクラスです。
     */
    class PlainSha256AdvertiseBuilder : public IAdvertiseBuilder
    {
    public:

        PlainSha256AdvertiseBuilder() NN_NOEXCEPT
            : m_Impl(AdvertiseFormat_PlainSha256, CalculateSha256)
        {
        }

        virtual ~PlainSha256AdvertiseBuilder() NN_NOEXCEPT
        {
        }

        virtual void SetNetworkId(const NetworkId& networkId) NN_NOEXCEPT NN_OVERRIDE
        {
            m_Impl.SetNetworkId(networkId);
        }

        virtual void SetVersion(Version version) NN_NOEXCEPT NN_OVERRIDE
        {
            m_Impl.SetVersion(version);
        }

        virtual void Build(
            void* pOutAdvertise, size_t* pOutSize, size_t bufferSize,
            const void* data, size_t dataSize) NN_NOEXCEPT NN_OVERRIDE
        {
            m_Impl.Build(pOutAdvertise, pOutSize, bufferSize, data, dataSize);
        }

    private:

        impl::PlainAdvertiseBuilderImpl m_Impl;
    };

    /**
     * @brief       平文の Advertise を解析するクラスです。
     */
    class PlainSha256AdvertiseParser : public IAdvertiseParser
    {
    public:

        static size_t GetRequiredBufferSize() NN_NOEXCEPT
        {
            return impl::PlainAdvertiseParserImpl::GetRequiredBufferSize();
        }

        PlainSha256AdvertiseParser(void* buffer, size_t bufferSize) NN_NOEXCEPT
          : m_Impl(buffer, bufferSize, AdvertiseFormat_PlainSha256, CalculateSha256)
        {
        }

        virtual ~PlainSha256AdvertiseParser() NN_NOEXCEPT
        {
        }

        virtual AdvertiseParserResult Parse(
            const void* data, size_t dataSize) NN_NOEXCEPT NN_OVERRIDE
        {
            AdvertiseParserResult result = m_Impl.VerifyHeader(data, dataSize);
            if (result == AdvertiseParserResult_Success)
            {
                result = m_Impl.VerifySign(data, dataSize);
                if (result == AdvertiseParserResult_Success)
                {
                    m_Impl.Parse(data, dataSize);
                }
            }
            return result;
        }

        virtual const NetworkId GetNetworkId() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Impl.GetNetworkId();
        }

        virtual const Version GetVersion() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Impl.GetVersion();
        }

        virtual size_t GetDataSize() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Impl.GetDataSize();
        }

        virtual const void* GetData() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Impl.GetData();
        }

    private:

        impl::PlainAdvertiseParserImpl m_Impl;
    };

}}} // namespace nn::ldn::detail
