﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/lcs/lcs_Types.h>
#include <nn/lcs/detail/lcs_ApplicationShareInfo.h>
#include <nn/lcs/detail/lcs_NetworkInfo.h>
#include <nn/lcs/detail/lcs_SystemShareInfo.h>
#include <nn/lcs/detail/Advertise/lcs_Advertise.h>


namespace nn { namespace lcs { namespace detail
{
    const int ApplicationRecordListSize = 16;

    // 分裂して受信者を待つまでの時間
    const int WaitReceiverTimeSeconds = 20;

    // システム更新のための再起動をしたときに戻ってくるまで待つ時間
    const int WaitRebootTimeSeconds = 1800;

    // タスクの完了を見に行く間隔
    const int AsyncWaitIntervalMilliSeconds = 1000;

    // 同じ相手に対してタスクをリトライする回数の最大値
    const int ShareTaskRetryCountMax = 5;

    // 同じ相手に対してタスクをリトライするまで待つ時間
    const int ShareTaskRetryIntervalMilliSeconds = 500;

    const size_t ApplicationDeliveryInfoHashSize = 32;

    enum LcsRole
    {
        LcsRole_Host = 0x01,
        LcsRole_Client,
        LcsRole_Fork,
    };

    enum RejectReason
    {
        RejectReason_None = 0x00,
        RejectReason_IncompatibleSystemVersion,
        RejectReason_ConnectionFailed,
        RejectReason_CommunicationError,
        RejectReason_NodeLeaved,
        RejectReason_CannotUpdateSystem,
        RejectReason_NotExistDownloadContents,
        RejectReason_UnshareableContents,
        RejectReason_Unknown,
        RejectReason_LowerSystemVersion,
        RejectReason_HigherSystemVersion,
    };

    enum LeaveReason
    {
        LeaveReason_None = 0x00,
        LeaveReason_Reboot,
    };

    enum SystemUpdateReason
    {
        SystemUpdateReason_IncompatibleContentsInfo = 0x01,
        SystemUpdateReason_NeedToRunApplication,
    };

    struct NodeEntryInfo
    {
        uint32_t index;
        char _reserve[4];
        AddressEntry nodeAddressEntry;
        char _reserve2[7];
        char userName[UserNameBytesMax + 1];
    };
    NN_STATIC_ASSERT(sizeof(NodeEntryInfo) == 160);

    struct MigrationInfo
    {
        LcsRole role;
        int nodeCount;
        char _reserve[56];
        NetworkSetting newNetworkSetting;
        NodeEntryInfo nodeEntryInfo[NodeCountMax];
    };
    NN_STATIC_ASSERT(sizeof(MigrationInfo) == 1408);

    const size_t MigrationInfoSize = sizeof(MigrationInfo);


    struct ApplicationDetailInfo
    {
        Bit64 id;
        Bit32 attributeFlag;
        char reserved1[4];
        int64_t requiredSize;
        char displayVersion[DisplayVersionSizeMax];
        char reserved2[88];
    };
    NN_STATIC_ASSERT(sizeof(ApplicationDetailInfo) == 128);

    struct ApplicationInfo
    {
        ApplicationDetailInfo detailInfo;
        int deliveryInfoCount;
        ApplicationDeliveryInfo deliveryInfo[ApplicationDeliveryInfoCountMax];
    };

    struct ShareComponent
    {
        uint8_t role;
        uint32_t index;
        Ipv4Address address;
        SystemDeliveryInfo systemDeliveryInfo;
        Bit64 id;
        nn::util::BitFlagSet<8, ContentsType> contentsFlag;
        ContentMetaKey contentMetaKey[ContentMetaKeyMax];
        int contentMetaKeyCount;
        int connectTimeout;
        int channel;
    };

    struct ApplicationDeliveryInfoHash
    {
        Bit64 id;
        char hash[ApplicationDeliveryInfoHashSize];
    };

    bool HasApplication(Bit64 id) NN_NOEXCEPT;

    Result GetApplicationDetailInfo(
        ApplicationDetailInfo* pOutInfo, void* buffer, size_t bufferSize, Bit64 id) NN_NOEXCEPT;

    Result GetApplicationControlData(size_t* pOutSize, void* buffer, size_t bufferSize, Bit64 id) NN_NOEXCEPT;

    int64_t GetRequiredSize(Bit64 id) NN_NOEXCEPT;

    bool HasFreeSpaceSize(int64_t needSize) NN_NOEXCEPT;

    void ConvertApplicationInfoToContetsInfo(
        ContentsInfo *pOutInfo, const ApplicationDetailInfo& appInfo) NN_NOEXCEPT;

    Result GetApplicationDeliveryInfoHash(
        void *pHash, size_t bufferSize, const ApplicationInfo &appInfo) NN_NOEXCEPT;

    bool IsApplicationDeliveryInfoHash(const ApplicationDeliveryInfoHash& appHash) NN_NOEXCEPT;

}}} // end of namespace nn::lcs
