﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief API declaration for PositionReader Processor
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/irsensor/irsensor_Result.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_MarkerPositionReaderTypes.h>


namespace nn { namespace irsensor {

/**
 * @name Marker Position Reader
 * @{
 */

/**
 *  @brief                Start MarkerPositionReader mode of IR sensor
 *  @param[in] handle     Handle of IR camera
 *  @return               Return processed result
 *  @pre
 *                      - Camera handle is already initialized via Initialize API
 *                      - For given handle, IR camera is not running yet
 *  @details            User sets a handle to PositionReader and IR sensor starts processing.
 *                      User keeps getting data via Get**() API while PositionReader is running.
 *                      When IR sensor is already running another mode, it stops running mode and starts MarkerPositionReader mode.
 */
::nn::Result RunMarkerPositionReader(const IrCameraHandle& handle) NN_NOEXCEPT;


/**
 *  @brief                           Retrieve latest processed data by PositionReader
 *
 *  @param[out] pOutValueArray       User allocated area that will receive the processed data
 *  @param[out] pReturnCount         Number of elements written to pOutValueArray (starting at index 0)
 *  @param[in]  countMax             Maximum number of elements that can be written to pOutValueArray
 *  @param[in]  handle               Handle of IR sensor
 *
 * @details                          This function retrieves PositionReader states calcuted by IR sensor data.
 *                                   The function will return all the states available available states.
 *
 *                                   Compared with getting other irsensor state like in Moment or Clustering mode,
 *                                   getting PositionReader state will not only copying value coming from irsensor
 *                                   but also calculating it which is not negligible.
 *
 *                                   When IR sensor is in trouble and unable to use, internal system displays the error.
 *                                   While displaying, this API blocks the thread which calls it.
 *                                   After user handles the error display, it returns nn::irsensor::ResultIrsensorDeviceError .
 *
 *                                   When the controller firmware is needed to update,
 *                                   internal system calls controller support applet and updates it.
 *                                   While updating, this API blocks the thread which calls it.
 *                                   If it succeeds it returns nn::irsensor::ResultIrsensorNotReady ,
 *                                   and if it fails it returns nn::irsensor::ResultIrsensorDeviceError .
 *
 *                                   IR sensor is not available when application uses NFP functions on at least one controller.
 *                                   The second IR sensor is not available only when the total number of connected controller is larger than 4.
 *                                   If these constraints are not satisfied, this API returns nn::irsensor::ResultIrsensorNotReady.
 *
 *  @retresult
 *   @handleresult{nn::irsensor::ResultIrsensorNotReady}
 *   @handleresult{nn::irsensor::ResultIrsensorUnavailable}
 *   @handleresult{nn::irsensor::ResultIrsensorDeviceError}
 *  @endretresult
 *  @pre
 *                         - Handle is already initialized via Initialize API.
 *                         - For given handle, IR camera is running as PositionReader
 *                         - pOutValueArray != nullptr
 *                         - pReturnCount != nullptr
 *  @post
 *                         - For all pOutValue in pOutValueArray .. pOutValueArray + pReturnCount - 1:
 *                             - 0 <= pOutValue->position.x && pOutValue->position.x <= MarkerPositionWidth
 *                             - 0 <= pOutValue->position.y && pOutValue->position.y <= MarkerPositionHeight
 *                             - 0.0f <= pOutValue->touchFactor && pOutValue->touchFactor <= 1.0f
 */

::nn::Result GetMarkerPositionReaderStates(MarkerPositionReaderState* pOutValueArray, int *pReturnCount, int countMax, const IrCameraHandle& handle) NN_NOEXCEPT;

/**
 * @}
 */

} } // namespace nn::irsensor
