﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Struct declaration for HeartRate Processor
 */

#pragma once

#include <nn/nn_TimeSpan.h>

namespace nn { namespace irsensor {

const int HeartRateProcessorWorkBufferSize = 5184; //!< Workbuffer size required for heartrate processing

/**
 * @brief Workbuffer used in heartrate processing
 */
struct HeartRateProcessorWorkBuffer
{
    uint8_t buffer[HeartRateProcessorWorkBufferSize];
};

/**
 * @brief Status of heartrate measurement
 */
enum HeartRateStatus
{
    HeartRateStatus_Invalid,                        //!< invalid result.
    HeartRateStatus_WaitingForUser,                 //!< waiting user to lay down his finger to start measurement.
    HeartRateStatus_MeasurementInProgress,          //!< measurement is in progress and minimum duration time hasn't elapsed yet.
    HeartRateStatus_MeasurementInProgressExtension, //!< regular duration time has elapsed we entered extension time.
    HeartRateStatus_MeasurementCompleted,           //!< measurement is done.
    HeartRateStatus_MeasurementAborted,             //!< measurement aborted because user took off his finger while measurement wasn't complete.
};

/**
 * @brief Structure for heartrate processor configuration
 */
struct HeartRateProcessorConfig
{
    ::nn::TimeSpan measurementDuration; //!< minimum duration of the heartrate measurement. capture will never complete before the end of this duration, whether a BPM is found early or not. At the end of this duration, if a BPM is found with perfect confidence: capture is complete.
    ::nn::TimeSpan measurementDurationExtension; //!< this extension time is added to measurement duration. capture ends when this amount has elapsed, or as soon as a BPM is found with perfect confidence.
};

/**
 * @brief Structure for storing heartrate processor state
 */
struct HeartRateProcessorState
{
    HeartRateStatus status; //!< status of the capture: informs whether BPM measurement has started, is in progress, completed...
    int64_t samplingNumber; //!< sampling number that increments when new data arrives to system (more than 0).
    uint32_t userId; //!< user id of the capture: this value increments when the user take off his finger.
    float progress; //!< value between [0, 1]: indicates the progression percentage of the measurement. it starts when user finger is layed down and stable on the sensor. If extension time is triggered, it stops at 0.99 and reaches 1 when a BPM with perfect confidence is found or when extension time is elapsed.
    float bpm; //!< BPM computed so far. Value is 0 if BPM isn't found yet.
    float confidence; //!< Value betwen [0, 1]: confidence percentage in the BPM value. this should always be checked to know whether the given BPM is trustable or not. Actually, this confidence value is tuned such as: BPM is correct and capture could end/success when the confidence percentage is >= 60%.
    float signalQualityInstant; //!< value between [0, 1]: signal quality in percentage and should be close to 1 to know how good the user is holding the sensor over latest short period.
    float signalQualityAverage; //!< value between [0, 1]: signal quality in percentage and should be close to 1, to know how good the user is holding the sensor over the whole measurement duration.
    int32_t newSignalValuesCount;  //!< for displaying signal in the user interface: array number of new signal values.
    float const *newSignalValues;  //!< for displaying signal in the user interface: array of new signal values. this pointer points to user workbuffer and is updated by calling GetHeartRateProcessorState(). this is only the new values, values given in previous states are not kept, so make sure to save them.
};

}} // namespace nn::irsensor

