﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Npad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace system {

struct NpadSystemId
{
    static const NpadIdType Other = 0x10;       //!< その他の本体機能操作用の Npad
};

typedef ::nn::util::BitFlagSet<32, NpadStyleTag>::Flag<30> NpadStyleSystem;          //!< 本体機能操作を表す Npad の種類です。

/**
 * @brief       Npad の操作の入力状態を表す構造体です。
 *
 * @details     本体機能を操作するのに必要な必要最小限の入力状態(十字, A, B, X, Y, L, R) を提供します。
 *              その他の入力状態は無入力 ( デジタルボタンの押下は無く、アナログスティックはニュートラル位置 ) となります。
 */
struct NpadSystemState
{
    int64_t samplingNumber;             //!< Npad の入力状態が更新される度に増加する値です。
    NpadButtonSet buttons;              //!< Npad のデジタルボタンの状態です。
    AnalogStickState analogStickL;      //!< Npad の左アナログスティックの状態です。
    AnalogStickState analogStickR;      //!< Npad の右アナログスティックの状態です。
    NpadAttributesSet attributes;       //!< Npad の状態を表す属性値です。
};

//! @name Npad 関連 API
//! @{

/**
 * @brief       Npad の本体機能操作の最新の入力状態を取得します。
 *
 * @details     指定の Npad ID と対応する Npad から GetNpadStates() で 1 つの入力状態を読み出した時と同じ値が返ります。
 *              最低 1 回の入力状態の更新が保証されるため、 SetSupportedNpadIdType() 呼び出し完了直後から利用可能です。
 *
 * @param[out]  pOutValue                   入力状態を読み出すバッファ
 * @param[in]   id                          Npad ID
 *
 * @pre
 *              - 指定の Npad ID が SetSupportedNpadIdType() で有効化されている
 *              - pOutValue != nullptr
 * @post
 *              - !(pOutValue->buttons.Get<NpadButton::Left>() && pOutValue->buttons.Get<NpadButton::Right>())
 *              - !(pOutValue->buttons.Get<NpadButton::Up>() && pOutValue->buttons.Get<NpadButton::Down>())
 */
void GetNpadState(NpadSystemState* pOutValue, const NpadIdType& id
                  ) NN_NOEXCEPT;

/**
 * @brief       Npad の本体機能操作の入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、指定の Npad ID と対応する Npad から入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *
 *              読み出し可能な入力状態の最大数は NpadStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は NpadSystemState::samplingNumber を参照してください。
 *
 *              取得できる入力は、十字ボタン, ABXY, LR のみです。
 *              十字ボタンのないデバイスについては、アナログスティックの入力で十字ボタンをエミュレーションします。
 *
 *              指定の Npad ID が本体機能操作が有効になっていない場合、入力状態は無入力（デジタルボタンの押下は無し）となります。
 *
 * @param[out]  pOutValues                  入力状態を読み出すバッファ（配列）
 * @param[in]   count                       読み出す入力状態の数
 * @param[in]   id                          Npad ID
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - 指定の Npad ID が SetSupportedNpadIdType() で有効化されている
 *              - pOutValues != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 読み出した入力状態 pOutValues[i] について
 *                  - !(pOutValues[i].buttons.Get<NpadButton::Left>() && pOutValues[i].buttons.Get<NpadButton::Right>())
 *                  - !(pOutValues[i].buttons.Get<NpadButton::Up>() && pOutValues[i].buttons.Get<NpadButton::Down>())
 */
int GetNpadStates(NpadSystemState* pOutValues,
                  int count,
                  const NpadIdType& id) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid
