﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Npad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/system/hid_CommonTypes.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_UniquePad.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       Npad のデバイスの種類です
 */
typedef DeviceType NpadDeviceType;

/**
 * @brief       Npad のデバイスの種類の集合を扱う型です。
 */
typedef DeviceTypeSet NpadDeviceTypeSet;

/**
 * @brief       Npad の電池残量を表す型です。
 */
typedef BatteryLevel NpadBatteryLevel;

const NpadBatteryLevel NpadBatteryLevel_None        = BatteryLevel_None;           //!< 電池がない
const NpadBatteryLevel NpadBatteryLevel_CriticalLow = BatteryLevel_CriticalLow;    //!< 電池残量 Critical Low
const NpadBatteryLevel NpadBatteryLevel_Low         = BatteryLevel_Low;            //!< 電池残量 Low
const NpadBatteryLevel NpadBatteryLevel_Midium      = BatteryLevel_Medium;         //!< 電池残量 Medium
const NpadBatteryLevel NpadBatteryLevel_High        = BatteryLevel_High;           //!< 電池残量 High

/**
 * @brief      Npad の電源状態を表す型です。
 */
typedef PowerInfo NpadPowerInfo;

//! @name Npad 本体機能関連 API
//! @{

/**
 * @brief       Npad デバイスの種類を取得します。
 *
 * @details     Npad Id で指定された Npad について、物理的なデバイスの種類を取得します。
 *              Npad では1つのNpad Id に対して複数のデバイスが接続される可能性もあるため、デバイスの種別は複数存在する可能性があります。
 *              接続されているデバイスに応じた種別が取得されるため、何も接続されていないときは返り値 n について n.IsAllOff() == true となります。
 *              未知のデバイスまたは種別が判定できないデバイスは NpadDeviceType::Unknown が返ります。
 *
 * @param[in]   id                      Npad ID
 *
 * @return      Npad のデバイスの種類
 */
NpadDeviceTypeSet GetNpadDeviceType(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Npad の電池の有無を取得します。
 *
 * @details     Npad Id で指定された Npad について、電池の有無を取得します。
 *              Npad に対して左右2本の Joy-Con が接続されている場合、どちらの状態が取得できるかは不定です。
 *
 * @param[in]   id                      Npad ID
 *
 * @return      Npad の電池の有無
 */
bool HasBattery(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Joy-Con が接続された Npad の電池の有無を取得します。
 *
 * @details     Npad Id で指定された Npad について、電池の有無を取得します。
 *              Npad に対して Joy-Con が接続されているときに有効です。
 *              Npad に Joy-Con 以外のデバイスが接続されている場合は、何もおこないません。
 *
 * @param[out]  pOutLeftJoy             Joy-Con (L) の電池の有無
 * @param[out]  pOutRightJoy            Joy-Con (R) の電池の有無
 * @param[in]   id                      Npad ID
 *
 * @pre
 *              - pOutLeftJoy != nullptr
 *              - pOutRightJoy != nullptr
 */
void HasBattery(bool* pOutLeftJoy, bool* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Npad の電池残量を取得します。
 *
 * @details     Npad Id で指定された Npad について、電池残量や充電状態を含めて電源の状態を取得します。
 *              Npad に対して左右2本の Joy-Con が接続されている場合は、 Joy-Con (R) の電源残量を取得します。
 *              Npad に対して電池が搭載されていないコントローラが接続されている場合、電池残量は不定になります。
 *
 * @param[in]   id                      Npad ID
 *
 * @return      Npad の電源状態
 */
NpadPowerInfo GetNpadPowerInfo(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Joy-Con が接続された Npad の電池残量を取得します。
 *
 * @details     Npad Id で指定された Npad について、電池残量や充電状態を含めて電源の状態を取得します。
 *              Npad に対して Joy-Con が接続されているときに有効です。
 *              Npad に Joy-Con 以外のデバイスが接続されている場合は、何もおこないません。
 *              Npad に対して電池が搭載されていないコントローラが接続されている場合、電池残量は不定になります。
 *
 * @param[out]  pOutLeftJoy             Joy-Con (L) の電源状態
 * @param[out]  pOutRightJoy            Joy-Con (R) の電源状態
 * @param[in]   id                      Npad ID
 *
 * @pre
 *              - pOutLeftJoy != nullptr
 *              - pOutRightJoy != nullptr
 */
void GetNpadPowerInfo(NpadPowerInfo* pOutLeftJoy, NpadPowerInfo* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       廃止されました。 GetNpadPowerInfo() を使用してください。
 *
 */
NN_DEPRECATED
bool IsNpadCharging(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       廃止されました。 GetNpadPowerInfo() を使用してください。
 *
 */
NN_DEPRECATED
NpadBatteryLevel GetNpadBatteryLevel(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       廃止されました。 GetNpadPowerInfo() を使用してください。
 *
 */
NN_DEPRECATED
void IsNpadCharging(bool* pOutLeftJoy, bool* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       廃止されました。 GetNpadPowerInfo() を使用してください。
 *
 */
NN_DEPRECATED
void GetNpadBatteryLevel(NpadBatteryLevel* pOutLeftJoy, NpadBatteryLevel* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       指定した Npad に対して ABXY ボタンと UI 表示して問題ないかどうか取得します。
 *
 * @details     NpadStyleSystem や NpadStyleSystemExt では複数の種類のコントローラーや持ち方に対応しています。
 *              そのため、ABXYボタンからの入力であったとしても、ABXY と印字されてない可能性があります。
 *              ABXY とボタンが印字されている場合は UI で ABXY と表示し、
 *              それ以外の場合は、印字に依存しない表示 (クローバー表示) を行います。
 *
 * @param[in]   id                      Npad ID
 *
 * @return      ABXY 表示しても問題ない場合、true が返ります
 *
 */
bool IsAbxyButtonOriented(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       指定した Npad に対して SL/SR ボタンと UI 表示して問題ないかどうか取得します。
 *
 * @details     NpadStyleSystem や NpadStyleSystemExt では複数の種類のコントローラーや持ち方に対応しています。
 *              そのため、L/Rボタンからの入力であったとしても、L/R と印字されてない可能性があります。
 *              Joy-Con 横持ちの場合は、L/RではなくSL/SRボタンが使われるため、UI で SL/SR と表示します。
 *              それ以外の場合は、等しくL/R と表記を行います。
 *
 * @param[in]   id                      Npad ID
 *
 * @return      SL/SR 表示しても問題ない場合、true が返ります
 *
 */
bool IsSlSrButtonOriented(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       指定した Npad の Plus ボタンや Minus ボタンの有無を取得します。
 *
 * @details     NpadStyleSystem や NpadStyleSystemExt では複数の種類のコントローラーや持ち方に対応しています。
 *              そのため、Npad に割り当てられているコントローラーによって Plus ボタンや Minus ボタンが存在しない場合が存在します。
 *              本関数を用いることで、現在接続されているコントローラーの Plus / Minus の有無を取得することができます。
 *
 * @param[in]   id                      Npad ID
 * @param[out]  pOutPlusButton          Plus ボタンを備えているかどうか
 * @param[out]  pOutMinusButton         Minus ボタンを備えているかどうか
 *
 */
void GetPlusMinusButtonCapability(bool* pOutPlusButton, bool* pOutMinusButton, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Npad の本体機能操作に含まれないボタンの入力状態を取得します。
 *
 * @details     本体機能では、何かしらの操作に割り当てられているボタン以外のボタンが押された場合に、
 *              NG 音を鳴らす予定です。
 *              本関数では、本体機能操作 (NpadStyleSystem または NpadStyleSystemExt) では取得できないボタン以外のボタン入力状態を取得します。
 *              対応していないボタンが複数存在する場合は、1つでも入力があった場合に true を返します。
 *
 * @param[in]   id                      Npad ID
 * @param[in]   style                   未使用を判定するうえで基準とする操作形態
 *
 * @return      対応していないボタンからの入力があった場合に true
 *
 * @pre         - style == NpadStyleSystem::Mask || style == NpadStyleSystemExt::Mask
 *
 */
bool IsUnsupportedButtonPressed(const NpadIdType& id, const NpadStyleSet& style) NN_NOEXCEPT;

/**
 * @brief       据置モード中の携帯機コントローラーのボタン操作状態を取得します。
 *
 * @details     コントローラーサポートアプレット内で、据置モード中の携帯機コントローラーのボタン操作状態を取得し、Joy-Con の取り外しを促したりするために使用します。
 *              本関数は上記アプレット以外での利用は禁止です。
 *
 * @return      据置モード中に携帯機コントローラーのボタン操作があった場合に true
 *
 */
bool IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT;

/**
 * @brief       本体機能向けの Npad の動作設定を適用します。
 *
 * @details     Nx の本体機能では、共通で利用される動作ポリシーが存在します。
 *              本関数を呼ぶことで、本体機能のアプレットに対して共通の動作ポリシーを適用します。
 *
 *              【本体機能向け動作ポリシー】
 *              - 操作形態 (NpadStyleSet) はすべて有効
 *              - 有効な NpadIdType は、 NpadId::Handheld / NpadId::No1 / NpadId::No2 / NpadId::No3 / NpadId::No4 / NpadSystemId::Other
 *              - Joy-Con は、2本持ちで接続される
 *              - アプリケーションから遷移してきた場合は、既に接続済みのコントローラーについては特に状態や操作形態はいじらずにそのまま利用できる
 *              　たとえば NpadId::No5 にコントローラーが割り当てられていても操作は可能で、切断されることはない
 *
 */
void ApplyNpadSystemCommonPolicy() NN_NOEXCEPT;

/**
 * @brief       全てのコントローラーの接続を常時許可する本体機能向けの Npad の動作設定を適用します。
 *
 * @details     Nx の本体機能内の特定のシーンにおいて、最大数のコントローラー接続を許可する場合に使用します。
 *
 *              【本体機能向け動作ポリシー】
 *              - 操作形態 (NpadStyleSet) はすべて有効
 *              - 有効な NpadIdType は、 NpadId::Handheld / NpadId::No1 から NpadId::No8 / NpadSystemId::Other
 *              - Joy-Con は、2本持ちで接続される
 *              - アプリケーションから遷移してきた場合は、既に接続済みのコントローラーについては特に状態や操作形態はいじらずにそのまま利用できる
 *              　たとえば NpadId::No5 にコントローラーが割り当てられていても操作は可能で、切断されることはない
 *
 */
void ApplyNpadSystemCommonPolicyFull() NN_NOEXCEPT;

/**
 * @brief       SlSr ボタンが押されて接続された際に、1本持ちとして接続することを許可します
 *
 * @details     本機能はコントローラーサポートアプレット内での利用を想定しています。
 *              NpadStlyeJoyLeft か NpadStyleJoyRight が有効な状態かつ、NpadJoyHoldType_Horizontal が有効な場合に、
 *              Sl / Sr ボタンが両方が押されて接続されたコントローラーについては、1本持ち (NpadJoyAssignmentMode_Single) として接続します
 *
 */
void EnableAssigningSingleOnSlSrPress() NN_NOEXCEPT;

/**
 * @brief       SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能を無効化します
 *
 * @details     EnableAssigningSingleOnSlSrPress() で有効になった機能を無効にします。
 *
 */
void DisableAssigningSingleOnSlSrPress() NN_NOEXCEPT;

/**
 * @brief       プレイヤーランプの点滅を行うデバイスを設定します
 *
 * @details     NpadIdType とデバイスタイプで指定されたコントローラーについて、プレイヤーランプの点滅を開始します。
 *              本関数は電池残量低下のオーバーレイ通知開始時に呼ばれることを想定しています。
 *              点滅を終了するためには、デバイスタイプのフラグをアンセットしてください
 *
 *              コントローラーが接続された直後は LED は必ず点灯状態にリセットされます
 *
 * @param[in]   id                      Npad ID
 * @param[in]   deviceType              点滅を行うデバイスのデバイスタイプ
 *
 */
void SetNpadPlayerLedBlinkingDevice(const NpadIdType& id, const NpadDeviceTypeSet& deviceType) NN_NOEXCEPT;

/**
 * @brief       最後に操作された Npad を取得します
 *
 * @details     複数のコントローラーが接続状態にある際に、最後に操作されたコントローラーを取得します
 *              ボタンが押下されているかアナログスティックの入力状態が原点からはずれていることを操作状態にあるとみなします。
 *
 * @param[out]  pOutValue               最後に操作された Npad の Npad ID
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      NpadIdType の取得に成功しました。}
 *   @handleresult{nn::hid::system::ResultNoActiveNpad,    最後に操作された NpadIdType が存在しません。}
 * @endretresult
 *
 */
Result GetLastActiveNpadController(NpadIdType* pOutValue) NN_NOEXCEPT;

/**
 * @brief       Npad に割り当てられた UniquePad を取得します
 *
 * @details     Npad に割り当てられた UniquePad の Id を取得します。
 *              UniquePad は物理的なデバイスに対して 1つずつ Id が割り当てられます。
 *              一方で、Npad では複数の物理デバイスがひとつの NpadIdType に対して割り当てられることがあります。
 *              そのため、Npad に対応する UniquePad を取得する場合複数取得されることがあります。
 *
 * @param[out]  pOutValues               Npad に対応づけられた UniquePadId のリストを受け取るための配列
 * @param[in]   count                    pOutValues の長さ
 * @param[in]   npadId                   UniquePad を取得する Npad の NpadIdType
 *
 * @pre
 *              - pOutValues != nullptr
 *              - count <= system::UniquePadIdCountMax
 *              - count > 0
 *
 * @return      取得された UniquePad の数
 *
 */
int GetUniquePadsFromNpad(system::UniquePadId* pOutValues, int count, const NpadIdType& npadId) NN_NOEXCEPT;

/**
 * @brief       Npad の接続形式を取得します。
 *
 * @details     Npad Id で指定された Npad について、接続形式を取得します。
 *              Npad に対して左右2本の Joy-Con が接続されている場合、どちらの状態が取得できるかは不定です。
 *
 * @param[in]   id                      Npad ID
 *
 * @return      Npad の接続形式
 */
InterfaceType GetNpadInterfaceType(const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Joy-Con が接続された Npad の接続形式を取得します。
 *
 * @details     Npad Id で指定された Npad について、接続形式を取得します。
 *              Npad に対して Joy-Con が接続されているときに有効です。
 *              Npad に Joy-Con 以外のデバイスが接続されている場合は、何もおこないません。
 *
 * @param[out]  pOutLeftJoy             Joy-Con (L) の接続形式
 * @param[out]  pOutRightJoy            Joy-Con (R) の接続形式
 * @param[in]   id                      Npad ID
 *
 * @pre
 *              - pOutLeftJoy != nullptr
 *              - pOutRightJoy != nullptr
 */
void GetNpadInterfaceType(InterfaceType* pOutLeftJoy, InterfaceType* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Switch Pro Controller のグリップ色を取得します
 *
 * @details     Switch Pro Controller の左及び右のグリップ色を取得します。
 *              Switch Pro Controller 以外に対して本機能を呼び出した場合は、ResultNpadControllerNotConnected が返ります。
 *
 * @param[out]  pOutLeftGrip            左グリップの色
 * @param[out]  pOutRightGrip           右グリップの色
 * @param[in]   id                      Npad ID
 *
 * @pre
 *              - pOutLeftGrip != nullptr
 *              - pOutRightGrip != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           色の取得に成功しました。}
 *   @handleresult{nn::hid::ResultNpadControllerNotConnected,   Switch Pro Controller が接続されていません。}
 * @endretresult
 */
Result GetNpadFullKeyGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       コンサポを呼び出したアプリケーションもしくはアプレットが対応している Npad の操作形態を取得します。
 *
 * @details     コンサポ内で有効にする Npad の操作形態を判断するために、コンサポ呼び出し元のアプリが有効にしている Npad の操作形態を取得します。
 *              本関数はアプリケーション/アプレットからコンサポが呼び出された場合にのみ有効です
 *              LA から LA を呼び出した場合は、ApplyNpadSystemCommonStyle() で有効になる操作形態が返されます。
 *
 * @pre
 *              - アプリケーション/アプレットから呼び出されたアプレット内での呼び出しであること
 *
 * @return
 *              呼び出し元のアプリケーション/アプレットが有効にしている Npad の操作形態
 */
NpadStyleSet GetSupportedNpadStyleSetOfCallerApplet() NN_NOEXCEPT;


//! @}

}}} // namespace nn::hid::system
