﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ファームウェア更新に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace hid { namespace system {

/**
 * @brief       ファームウェアバージョンが紐付くデバイスを表す識別子の最大長です。
 */
const int FirmwareVersionIdentifierLengthMax = 4;

/**
 * @brief       ファームウェア更新が必要な理由です。
 */
enum FirmwareUpdateRequiredReason : uint8_t
{
    FirmwareUpdateRequiredReason_Nothing,       //!< 更新は必要ありません。
    FirmwareUpdateRequiredReason_Corrupted,     //!< ファームウェアが壊れており、復旧が必要な状態です。
    FirmwareUpdateRequiredReason_Hotfix         //!< 重要な不具合修正があります。
};

/**
 * @brief       ファームウェア更新中のステージを表す型です。
 */
enum FirmwareUpdateStage : uint8_t
{
    FirmwareUpdateStage_Preparing,          //!< ファームウェア更新の準備中です。
    FirmwareUpdateStage_Updating,           //!< ファームウェアの更新中です。
    FirmwareUpdateStage_Finalizing,         //!< ファームウェア更新の終了処理中です。
    FirmwareUpdateStage_Completed           //!< ファームウェアの更新が完了しました。
};

/**
 * @brief       ファームウェアバージョンです。
 */
struct FirmwareVersion
{
    uint8_t major;          //!< ファームウェアのメジャーバージョンです。
    uint8_t minor;          //!< ファームウェアのマイナーバージョンです。
    uint8_t micro;          //!< ファームウェアのマイクロバージョンです。
    uint8_t revision;       //!< ファームウェアのリビジョン番号です。

    char    identifier[FirmwareVersionIdentifierLengthMax];     //!< デバイスの種類を示す識別子です。ヌル終端の文字列が格納されます。

    char    _reserve[8];
};

/**
 * @brief       ファームウェア更新を実行しているデバイスに対するハンドルです。
 */
struct FirmwareUpdateDeviceHandle
{
    uint64_t _storage;
};

/**
 * @brief       ファームウェア更新の進捗状況です。
 */
struct FirmwareUpdateState
{
    FirmwareUpdateStage stage;          //!< 実行中のファームウェア更新の内容です。 @ref FirmwareUpdateStage の値が入ります。
    uint8_t             progress;       //!< ファームウェア更新全体の進捗率です。 0-100 の値で取得されます。

    char                _reserve[2];
};

//! @name ファームウェア更新 関連 API
//! @{

/**
 * @brief       ファームウェア更新機能を初期化します。
 *
 * @details     ファームウェア更新に必要な初期化処理を行います。
 *              ファームウェア更新を開始する前に、必ず本関数を呼び出す必要があります。
 */
void InitializeFirmwareUpdate() NN_NOEXCEPT;

/**
 * @brief       コントローラーのファームウェアバージョンを取得します。
 *
 * @details     指定したコントローラーのファームウェアバージョンを取得します。
 *
 *              まだファームウェアバージョンを取得していないコントローラーに対して本 API を呼び出すと、バージョンの読み出しを開始します。
 *              ファームウェアバージョンの読み出し処理には時間がかかるため、読み出し中は @ref ResultFirmwareVersionReading を返します。
 *              @ref ResultFirmwareVersionReading が返った場合は、少なくとも 15 msec 以上経過してからリトライしてください。
 *              一度ファームウェアバージョンの読み出しが完了した後は、コントローラーが切断されるまではすぐにバージョンを取得できます。
 *
 * @param[out]  pOutVersion                 読み出されたファームウェアバージョン
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @pre
 *  - @a pOutVersion != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                       処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,            指定した UniquePadId にコントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareVersionReading,           ファームウェアバージョンを読み出し中です。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateCorrupted,          ファームウェアが壊れています。復旧するにはファームウェアを更新する必要があります。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateHardwareError,      コントローラーにハードウェア異常が発生しました。コントローラーをリセットしてください。繰り返し発生する場合は修理が必要です。}
 * @endretresult
 */
::nn::Result GetFirmwareVersion(FirmwareVersion* pOutVersion,
                                UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       更新先ファームウェアのバージョンを取得します。
 *
 * @details     指定したコントローラーに対応する、システムデータ内のファームウェアバージョンを取得します。
 *
 * @param[out]  pOutVersion                 更新先ファームウェアのバージョン
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @pre
 *  - @a pOutVersion != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,    指定した UniquePadId にコントローラーが接続されていません。}
 * @endretresult
 */
::nn::Result GetAvailableFirmwareVersion(FirmwareVersion* pOutVersion,
                                         UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       更新可能な最新ファームウェアが存在するか確認します。
 *
 * @details     指定したコントローラーに、更新可能なファームウェアが存在するか確認します。
 *              コントローラーのファームウェアが壊れていて復旧が必要な場合も、更新可能なファームウェアが存在するものとして扱われます。
 *
 *              まだファームウェアバージョンを取得していないコントローラーに対して本 API を呼び出すと、バージョンの読み出しを開始します。
 *              ファームウェアバージョンの読み出し処理には時間がかかるため、読み出し中は @ref ResultFirmwareVersionReading を返します。
 *              @ref ResultFirmwareVersionReading が返った場合は、少なくとも 15 msec 以上経過してからリトライしてください。
 *              一度ファームウェアバージョンの読み出しが完了した後は、コントローラーが切断されるまではすぐに結果を取得できます。
 *
 * @param[out]  pOutIsAvailable             更新可能ファームウェア有無の格納先
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @pre
 *  - @a pOutIsAvailable != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                       処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,            指定した UniquePadId にコントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareVersionReading,           ファームウェアバージョンを読み出し中です。読み出し中は、新しいバージョンの有無を確定できません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateHardwareError,      コントローラーにハードウェア異常が発生しました。コントローラーをリセットしてください。繰り返し発生する場合は修理が必要です。}
 * @endretresult
 */
::nn::Result IsFirmwareUpdateAvailable(bool* pOutIsAvailable,
                                       UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新が必要か確認します。
 *
 * @details     指定したコントローラーに対して、ファームウェア更新が必要な状態であるか確認します。
 *
 *              最新ファームウェアが存在していても、全コントローラーに対して更新が必須でなければ、本 API では更新不要として扱われる場合があります。
 *              単純に最新ファームウェアへの更新有無を判定する場合は、 @ref IsFirmwareUpdateAvailable を使用してください。
 *
 *              @ref FirmwareUpdateRequiredReason は、将来定義が追加される可能性があります。
 *              @a pOutReason に現状未定義の値が格納された場合でも、動作に問題がないように実装してください。
 *              @ref FirmwareUpdateRequiredReason_Nothing 以外の値が格納された場合は、更新が必要な状態です。
 *
 *              まだファームウェアバージョンを取得していないコントローラーに対して本 API を呼び出すと、バージョンの読み出しを開始します。
 *              ファームウェアバージョンの読み出し処理には時間がかかるため、読み出し中は @ref ResultFirmwareVersionReading を返します。
 *              @ref ResultFirmwareVersionReading が返った場合は、少なくとも 15 msec 以上経過してからリトライしてください。
 *              一度ファームウェアバージョンの読み出しが完了した後は、コントローラーが切断されるまではすぐに結果を取得できます。
 *
 * @param[out]  pOutReason                  確認結果の格納先
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @pre
 *  - @a pOutReason != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                       処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,            指定した UniquePadId にコントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareVersionReading,           ファームウェアバージョンを読み出し中です。読み出し中は、更新の要否を確定できません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateHardwareError,      コントローラーにハードウェア異常が発生しました。コントローラーをリセットしてください。繰り返し発生する場合は修理が必要です。}
 * @endretresult
 */
::nn::Result CheckFirmwareUpdateRequired(FirmwareUpdateRequiredReason* pOutReason,
                                         UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新の処理を開始します。
 *
 * @details     指定したコントローラーに対して、ファームウェア更新を開始します。
 *              ファームウェア更新中の進捗状況は、 @ref GetFirmwareUpdateState で取得できます。
 *
 *              @a pOutHandle に割り当てられたハンドルは、別のコントローラーのファームウェア更新開始時や、コントローラーの切断時に無効化されます。
 *
 *              同時にファームウェア更新ができるコントローラーは 1 台のみです。
 *              ファームウェア更新を行うためには、コントローラーの電池残量が @ref BatteryLevel_Low 以上、または充電中である必要があります。
 *
 *              ファームウェア更新中に有線接続・無線接続を切り替えた場合、更新は失敗します。
 *
 * @param[out]  pOutHandle                  ファームウェア更新を開始したコントローラーに対応するハンドル
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @pre
 *  - @ref InitializeFirmwareUpdate を実行済み
 *  - @a pOutHandle != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                       処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,            指定した UniquePadId にコントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateBusy,               別のコントローラーでファームウェア更新が行われています。}
 *   @handleresult{nn::hid::system::ResultFirmwareVersionReading,           ファームウェアバージョンを読み出し中です。バージョン読み出し中は更新内容が未確定なため、更新を開始できません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateNotNeeded,          既に最新のファームウェアに更新されています。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateLowBattery,         電池残量が少ないため、ファームウェア更新を開始できません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateHardwareError,      コントローラーにハードウェア異常が発生しました。コントローラーをリセットしてください。繰り返し発生する場合は修理が必要です。}
 * @endretresult
 */
::nn::Result StartFirmwareUpdate(FirmwareUpdateDeviceHandle* pOutHandle,
                                 UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新を中断します。
 *
 * @details     実行中のファームウェア更新を中断します。
 *              ファームウェア更新中でない場合は何も行いません。
 *
 *              ファームウェア更新中に中断した場合、一部機能が正常に動作しなくなる可能性があります。
 *              更新が中断されたコントローラーを復旧するには、再度ファームウェアを更新する必要があります。
 *
 * @pre
 *  - @ref InitializeFirmwareUpdate を実行済み
 */
void AbortFirmwareUpdate() NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新の経過状況を取得します。
 *
 * @details     ファームウェア更新の経過状況を取得します。
 *              本 API は、 @ref StartFirmwareUpdate で割り当てられたハンドルに対して呼び出してください。
 *
 *              ファームウェア更新の経過状況は、ステージと進捗率の 2 種類の値で取得されます。
 *
 *              ステージは、進行中のファームウェア更新処理の内容を示します。
 *
 *              進捗率は、ファームウェア更新の行程全体に対して、どの程度処理が完了したかを示します。
 *              値は 0 ～ 100 の値として取得されます。
 *
 * @param[out]  pOutState                   ファームウェア更新の経過状況
 * @param[in]   handle                      経過状況を取得するデバイスのハンドル
 *
 * @pre
 *  - @ref InitializeFirmwareUpdate を実行済み
 *  - @a pOutState != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                       処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,            コントローラーが接続されていないか、無効なハンドルが指定されました。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateNotStarted,         ファームウェア更新を開始していません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateFailed,             ファームウェアの更新に失敗しました。リトライで成功する可能性があります。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateHardwareError,      コントローラーにハードウェア異常が発生しました。コントローラーをリセットしてください。繰り返し発生する場合は修理が必要です。}
 * @endretresult
 */
::nn::Result GetFirmwareUpdateState(FirmwareUpdateState* pOutState,
                                    const FirmwareUpdateDeviceHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Hotfix 起因のファームウェア更新をスキップするように設定します。
 *
 * @details     isEnabled で指定された値にしたがって、@ref CheckFirmwareUpdateRequired が返す「ファームウェア更新が必要な理由」を変更します。
 *              isEnabled を True にすることで、
 *              FirmwareUpdateRequiredReason_Hotfix は FirmwareUpdateRequiredReason_Nothing に変更されます。
 *              FirmwareUpdateRequiredReason_Corrupted は変更されません。
 *
 *              このフラグのデフォルトは false です。
 *
 * @param[in]   isEnabled     更新機能をスキップする場合には true を、スキップしない場合は false を指定します。
 *
 */
void SetFirmwareHotfixUpdateSkipEnabled(bool isEnabled) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
