﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       オーディオコントロール に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace system {

const int AudioControlStateCountMax = 16; //!< 内部的に保持されるオーディオコントロールの入力状態の最大数

/**
 * @brief       オーディオコントロールの定義です。
 */
struct AudioControl
{
    typedef ::nn::util::BitFlagSet<64, AudioControl>::Flag<0> VolumeIncrement; //!< ボリュームアップ
    typedef ::nn::util::BitFlagSet<64, AudioControl>::Flag<1> VolumeDecrement; //!< ボリュームダウン
};

/**
 * @brief       オーディオコントロールの集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<64, AudioControl> AudioControlSet;


/**
 * @brief       オーディオコントロールの入力状態の定義です。
 */
struct AudioControlState
{
    int64_t samplingNumber;        //!< オーディオコントロールの入力状態が更新される度に増加する値です。
    AudioControlSet audioControls; //!< オーディオコントロールの入力状態です。
};

//! @name オーディオコントロール 関連 API
//! @{

/**
 * @brief       オーディオコントロール を初期化します。
 */
void InitializeAudioControl() NN_NOEXCEPT;

/**
 * @brief       オーディオコントロールの入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な入力状態の最大数は AudioControlStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は AudioControlState::SamplingNumber を参照してください。
 *
 * @param[out]  outStates                   入力状態を読み出すバッファ（配列）
 * @param[in]   count                       outStates の数
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - InitializeAudioControl() の呼び出しが完了している
 *              - outStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
int GetAudioControlStates(AudioControlState* outStates, int count) NN_NOEXCEPT;

/**
 * @brief        オーディオコントロールの入力状態変化とイベントを紐付けます。
 *
 * @param[in]    pEvent           オーディオコントロールの入力状態変化が発生した際にシグナル化するイベントを指定します。
 * @param[in]    clearMode        イベントオブジェクトのクリアモード
 *
 * @pre          - pEvent != nullptr
 *               - *pEvent は未初期化状態
 * @post         - *pEvent は初期化状態
 *               - *pEvent はシグナル状態
 */
void BindAudioControlStateChangeEvent(nn::os::SystemEventType* pEvent,
                                      nn::os::EventClearMode clearMode) NN_NOEXCEPT;

}}} // namespace nn::hid::system
