﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Seven 向け SixAxisSensor のセンサフュージョンに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>

namespace nn { namespace hid {

//! @name Seven 向け SixAxisSensor のセンサフュージョンに関する API
//! @{

/**
 * @brief       Seven 向け SixAxisSensor の加速度値による姿勢回転の補正強度を設定します。
 *
 * @details     Seven 向け SixAxisSensor が提供する姿勢回転 SevenSixAxisSensorState::rotation の値は、
 *              6 軸センサーから取得される加速度値と角速度値を相補的に利用して算出されます。
 *              本 API では、加速度による姿勢回転の補正強度を設定します。
 *              加速度値による補正処理は、内部的に推定された重力加速度をもとに、pitch、 roll 方向のずれを推定し姿勢回転を補正します。
 *              デフォルトの補正強度は 1.0f です。
 *
 *              アプリがインフォーカス状態の場合のみ本設定は有効になります。
 *              センサーのサンプリング開始直後や、アプリがインフォーカス状態ではない状態から復帰した直後など、
 *              姿勢回転のずれが起きる可能性がある場合に、素早く pitch、 roll 方向の姿勢回転を補正する目的での利用が考えられます。
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                処理に成功しました。}
 *   @handleresult{nn::ResultSevenSixAxisSensorNotSupported,         機能が対応していません。}
 * @endretresult
 *
 * @param[in]   strength                      加速度補正の重みです。値が大きいほど強い補正がかかります。
 *
 * @pre
 *              - strength >= 0.0f && strength <= 10.0f
 *              - InitializeSevenSixAxisSensor() の呼び出しが完了している
 */
Result SetSevenSixAxisSensorFusionStrength(float strength) NN_NOEXCEPT;

/**
 * @brief       Seven 向け SixAxisSensor の加速度値による姿勢回転の補正強度を取得します。
 *
 * @details     Seven 向け SixAxisSensor の加速度値による姿勢回転の補正強度を取得します。
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                処理に成功しました。}
 *   @handleresult{nn::ResultSevenSixAxisSensorNotSupported,         機能が対応していません。}
 * @endretresult
 *
 * @param[out]  pOutStrength                      加速度補正の重みです。値が大きいほど強い補正がかかります。
 *
 * @pre
 *              - pOutStrength != nullptr
 *              - InitializeSevenSixAxisSensor() の呼び出しが完了している
 */
Result GetSevenSixAxisSensorFusionStrength(float* pOutStrength) NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
