﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラの拡張バスに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace hid {

/**
 * @brief 拡張バスのハンドルです。
 */
struct ExternalBusHandle
{
    uint64_t _storage;
};

struct ExternalBusDeviceInfo
{
    uint32_t id;
};

/**
* @brief 拡張バスの種類です。
*/
enum ExternalBusType
{
    ExternalBusType_LeftJoyRail = 0,  //!< 左 Joy-Con のレール
    ExternalBusType_RightJoyRail = 1,  //!< 右 Joy-Con のレール
};

/**
* @brief Joy-Con で定期的に通信を行うモードの種類です。
*/
enum ExternalBusJoyPollingMode
{
    ExternalBusJoyPollingMode_SixAxisSensorDisable,   //!< Joy-Con の六軸センサを使用しないモードです。
    ExternalBusJoyPollingMode_SixAxisSensorEnable,    //!< Joy-Con の六軸センサと同時に使うモードです。
};

//! @name コントローラの拡張バス関連 API
//! @{

/**
* @brief       Npad に搭載されている拡張バスに接続されたデバイスを操作するためのハンドルを取得します。
*
* @details     Npad ID で指定された Npad に搭載されている拡張バスに接続されたデバイスを操作するためのハンドルを取得します。
               拡張バスに接続されたデバイスに対するハンドルは Npad Id に対して操作スタイル(NpadStyleSet)毎に取得します。
               操作スタイルは NpadStyleSet で指定しますが、複数の NpadStyle を同時に指定することはできません。
               NpadStyleSet によって取得できるハンドルの数が異なるため、取得できたハンドルの数が返り値として取得できます。
*
* @param[out]  pOutHandle      コントローラの拡張バスに接続されたデバイスのハンドルの格納先
* @param[in]   count           pOutHandle で取得するハンドルの最大数
* @param[in]   id              Npad ID
* @param[in]   style           ハンドルを取得する Npad の操作スタイル
*
* @return      取得されたハンドルの数
*
* @pre
*              - style.IsAnyOn() == true
*              - style.CountPopulation() == 1
*/
bool GetExternalBusHandle(ExternalBusHandle* pOutHandle, const nn::hid::NpadIdType &id, ExternalBusType busType) NN_NOEXCEPT;

/**
* @brief       ハンドルが有効か否かを返します。
*
* @details     ハンドルが有効か否かを返します。
*
* @param[in]  handle          拡張バスのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                           handle は有効です。}
*      @handleresult{nn::hid::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
*/
nn::Result CheckValidExternalBusHandle(ExternalBusHandle handle) NN_NOEXCEPT;

/**
* @brief       コントローラの拡張バスへデバイスが接続されているかどうかを取得します。
*
* @details     コントローラの拡張バスへデバイスが接続されているかどうかを取得します。
*              本機能では何らかのデバイスが接続されていることが取得できます。
*              接続されているデバイスの種類を取得したい場合は EnableExternalDevice() で有効化にした後に、
*              GetExternalDeviceInfo() を呼んでデバイスの種類を取得してください。
*
* @param[out]  pOutIsConnected 拡張バスへデバイスが接続されているかどうか
* @param[in]   handle          拡張バスのハンドル
*
* @return      拡張バスへデバイスが接続されているかどうか
*
* @pre
*              - 指定の handle が GetBusHandles() 関数の呼び出しによって取得されている
*/
nn::Result IsExternalBusDeviceConnected(bool* pOutIsConnected, const ExternalBusHandle& handle) NN_NOEXCEPT;

/**
 * @brief       コントローラの拡張バス機能を初期化します。
 *
 * @details     コントローラの拡張バス機能を初期化します。
 *              本 API 呼び出し後、依存がある一部の機能は使用できなくなります。
 *              本 API の呼び出しでは拡張バスに接続されたデバイスに対する操作は発生しません。
 *
 * @param[in]   handle          拡張バスのハンドル
 */
nn::Result InitializeExternalBus(const ExternalBusHandle& handle) NN_NOEXCEPT;

/**
* @brief       コントローラの拡張バス機能を終了します。
*
* @details     コントローラの拡張バス機能を終了します。
*              本 API 呼び出し後、依存がある一部の機能が使用できなるようになります。
*              本 API の呼び出しでは拡張バスに接続されたデバイスに対する操作は発生しません。
*
* @param[in]   handle          拡張バスのハンドル
*/
nn::Result FinalizeExternalBus(const ExternalBusHandle& handle) NN_NOEXCEPT;

/**
 * @brief       コントローラの拡張バスに接続されたデバイスの有効化・無効化を切り替えます。
 *
 * @details     コントローラの拡張バスに接続されたデバイスの有効化・無効化を切り替えます。
 *              本機能は拡張バスへ接続されたデバイスとの通信準備が完了するまでブロッキングします。
 *
 * @param[in]   handle          拡張バスのハンドル
 * @param[in]   isEnabled       有効化・無効化
 * @param[in]   version         コントローラの要求バージョン
 *
 * @pre
 *              Initialize() の呼び出しが完了している
 */
nn::Result EnableExternalBusDevice(const ExternalBusHandle& handle, bool isEnabled, uint64_t version) NN_NOEXCEPT;

/**
* @brief       コントローラの拡張バスへ接続されたデバイスの情報を取得します。
*
* @details     コントローラの拡張バスへ接続されたデバイスの情報を取得します。
*
* @param[in]   pOutInfo              拡張バスへ接続されたデバイスの情報
* @param[in]   handle                拡張バスのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                           処理に成功しました。}
*      @handleresult{nn::xcd::ResultNotConnected,                 デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotAttached,  拡張デバイスが接続されていません。}
*      @handleresult{nn::xcd::ResultAttachmentDeviceNotReady,     拡張デバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalBusDevice() で拡張デバイスが有効になっている必要があります。
*/
nn::Result GetExternalBusDeviceInfo(ExternalBusDeviceInfo* pOutInfo, const ExternalBusHandle& handle) NN_NOEXCEPT;

/**
* @brief       コントローラの拡張バスへ接続されたデバイスへコマンドを非同期に送信します。
*
* @details     コントローラの拡張バスへ接続されたデバイスへコマンドを非同期に送信します。
*              本関数は送信データを内部のキューへコピーすると返ります。
*              送信結果の受信完了は SetSendCommandAsycResultEvent() を使用してイベントを登録して待ってください。
*              イベントのシグナル後、 GetSendCommandAsynceResult() で結果を取得できます。
*
* @param[in]   pInCommand              　送信するコマンドが入ったバッファへのポインタ
* @param[in]   inCommandSize             送信するコマンドのサイズ
* @param[in]   handle                    コマンドを送信する拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                                 処理に成功しました。}
*      @handleresult{nn::hid::ResultExternalBusDeviceNotAttached,    拡張バスにデバイスが接続されていません。}
*      @handleresult{nn::hid::ResultExternalBusDeviceNotReady,       拡張バスに接続されたデバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalDevice() で拡張デバイスが有効になっている必要があります。
*/
nn::Result SendCommandAsyncToExternalBusDevice(const uint8_t* pInCommand, size_t inCommandSize, const ExternalBusHandle& handle) NN_NOEXCEPT;

/**
* @brief       コントローラの拡張バスへ接続されたデバイスへの SendCommand の Result を取得します。
*
* @details     コントローラの拡張バスへ接続されたデバイスへの SendCommand の Result を取得します。
*
*
* @param[in]   pOutSize              　 受信済み Result をコピーしたサイズ
* @param[in]   pOutResultBuffer         受信済み Result をコピーするバッファへのポインタ
* @param[in]   outResultBufferSize      受信済み Result をコピーするバッファのサイズ
* @param[in]   handle                   Result をコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                                 処理に成功しました。}
*      @handleresult{nn::hid::ResultExternalBusDeviceNotAttached,    拡張バスにデバイスが接続されていません。}
*      @handleresult{nn::hid::ResultExternalBusDeviceNotReady,       拡張バスに接続されたデバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalDevice() で拡張デバイスが有効になっている必要があります。
*/
nn::Result GetSendCommandAsynceResultFromExternalBusDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, const ExternalBusHandle& handle) NN_NOEXCEPT;

/**
* @brief       コントローラの拡張バスへ接続されたデバイスからのデータ送信結果の受信を通知するイベントを登録します。
*
* @details     コントローラの拡張バスへ接続されたデバイスからのデータ送信結果の受信を通知するイベントを登録します。
*
* @param[in]   pAttachmentDataReceiveEvent コントローラの拡張バスへ接続されたデバイスからのデータ送信結果の受信を通知するイベントへのポインタ
* @param[in]   handle                      イベントを登録する拡張デバイスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hid::ResultExternalBusDeviceNotAttached,    拡張バスにデバイスが接続されていません。}
*      @handleresult{nn::hid::ResultExternalBusDeviceNotReady,       拡張バスに接続されたデバイスとの通信ができない状態です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalDevice() で拡張デバイスが有効になっている必要があります。
*/
nn::Result BindEventForExternalBusDeviceSendCommandAsycResult(nn::sf::Out<nn::sf::NativeHandle> receiveEventHandle, const ExternalBusHandle& handle) NN_NOEXCEPT;


/**
* @brief       Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを有効化します。
*
* @details     Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを有効化します。
*              本モードを有効化すると、定期的にデバイスとの送受信を行います。
*              Joy-Con には六軸センサを同時に使うモードと同時に使わないモードがあります。
*              六軸センサと同時に使うモードの場合、定期受信できるサイズの最大値は 9 byte、読み出し可能な入力状態の最大数は 10 です。
*              六軸センサを使用しないモードの場合、定期受信できるサイズの最大値は 39 byte、読み出し可能な入力状態の最大数は 10 です。
*
* @param[in]   pInCommand                デバイスへ定期送信するデータが入ったバッファへのポインタ
* @param[in]   inCommandSize             デバイスへ定期送信するデータが入ったバッファのサイズ
* @param[in]   handle                    コマンドを送信する拡張バスへのハンドル
* @param[in]   mode                      ポーリングのモード
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hid::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hid::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hid::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalBusDevice() の呼び出しが完了している
*  - handle 取得時に busTypeId に ExternalBusLeftJoyConRailBusTypeId または ExternalBusRightJoyConRailBusTypeId を指定している。
*/
nn::Result EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, const ExternalBusHandle& handle, ExternalBusJoyPollingMode mode) NN_NOEXCEPT;

/**
* @brief       Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを無効化します。
*
* @details     Joy-Con レールに接続された拡張デバイスと定期的にデータ通信を行うモードを無効化します。
*
* @param[in]   handle                    コマンドを送信する拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hid::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hid::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hid::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している
*  - EnableExternalBusDevice() の呼び出しが完了している
*  - handle 取得時に busTypeId に ExternalBusLeftJoyConRailBusTypeId または ExternalBusRightJoyConRailBusTypeId を指定している。
*/
nn::Result DisablePollingReceiveModeForAttachmentDevice(const ExternalBusHandle& handle) NN_NOEXCEPT;


/**
* @brief       Joy-Con レールに接続された拡張デバイスと定期的に通信を行うモードの最新の受信データを取得します。
*
* @details     Joy-Con レールに接続された拡張デバイスと定期的に通信を行うモードの最新の受信データを取得します。
*
* @param[in]   pOutSize            受信データをコピーしたサイズ
* @param[in]   pOutCommand         受信データをコピーするバッファへのポインタ
* @param[in]   outCommandSize      受信データをコピーするバッファのサイズ
* @param[in]   handle              受信データをコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hid::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hid::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - Initialize() の呼び出しが完了している。
*  - EnableExternalDevice() の呼び出しが完了している
*  - EnablePollingReceiveMode() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetPollingDataForAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, const ExternalBusHandle& handle) NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
