﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       UniquePad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/detail/hid_PadTypes.h>
#include <nn/hid/detail/hid_SensorTypes.h>
#include <nn/hid/detail/hid_RxPacketHistoryTypes.h>

namespace nn { namespace hid { namespace debug {

const int UniquePadStateCountMax = 16;   //!< 内部的に保持される UniquePad の入力状態の最大数

//! @name UniquePad のデバッグ用機能
//! @{

/**
 * @brief       UniquePad のデジタルボタン及びアナログスティックの入力状態を読み出します、
 *
 * @details     指定の UniquePadId と対応する UniquePad から入力状態を読み出します。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は PadState::samplingNumber を参照してください。
 *
 *              ドライバレベルの未加工の値を返すため、禁則処理や正規化などの処理は行われません。
 *
 * @param[out]  pOutValue               入力状態を読み出すバッファ
 * @param[in]   id                      UniquePadId
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetUniquePadDriverState(detail::PadDriverState* pOutValue,
                            const system::UniquePadId& id) NN_NOEXCEPT;


/**
 * @brief       UniquePad の 6軸センサーの入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、指定の UniquePadId と対応する 6 軸センサーから入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              読み出し可能な入力状態の最大数は UniquePadStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は SixAxisSensorState::samplingNumber を参照してください。
 *
 *              読み出されたセンサー値はドライバレベルでパースされただけの生の値となります。
 *              較正値などによる補正処理は行われません。
 *
 * @param[out]  pOutStates              入力状態を読み出すバッファ（配列）
 * @param[in]   count                   読み出す入力状態の数
 * @param[in]   id                      UniquePadId
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - pOutStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
int GetSixAxisSensorDriverStates(detail::SixAxisSensorDriverState* pOutStates,
                                 int count,
                                 const system::UniquePadId& id) NN_NOEXCEPT;

/**
 * @brief       UniquePad の受信パケットの受信履歴を取得します。
 *
 * @details     指定の UniquePadId と対応する UniquePad から受信パケットの受信履歴を取得します。
 *              利用可能な受信履歴には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は RxPacketHistory::samplingCount を参照してください。
 *
 * @param[out]  pOutValue               入力状態を読み出すバッファ
 * @param[in]   id                      UniquePadId
 *
 * @pre
 *              - pOutValue != nullptr
 */
void GetRxPacketHistory(detail::RxPacketHistory* pOutValue,
                        const system::UniquePadId& id) NN_NOEXCEPT;


::nn::Result BindSerialFlashEventHandle(
    ::nn::os::SystemEventType* pEvent,
    ::nn::os::EventClearMode clearMode,
    const system::UniquePadId& id) NN_NOEXCEPT;

::nn::Result ReadSerialFlash(
    const uint32_t address,
    uint8_t* pOutBuffer,
    int size,
    const system::UniquePadId& id) NN_NOEXCEPT;

::nn::Result WriteSerialFlash(
    const uint32_t address,
    const uint8_t* pBuffer,
    int bufferSze,
    int writeSize,
    const system::UniquePadId& id) NN_NOEXCEPT;

::nn::Result GetSerialFlashResult(const system::UniquePadId& id) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::debug
