﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       TouchScreen のデバッグ機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>
#include <nn/hid/hid_TouchScreen.h>

namespace nn { namespace hid { namespace debug {

/**
 * @brief       TouchScreen の自動操作状態を表す構造体です。
 *
 * @tparam      N                           格納するタッチの数
 *
 * @pre
 *              - 0 < N && N <= TouchStateCountMax
 */
template<size_t N>
struct TouchScreenAutoPilotState
{
    NN_STATIC_ASSERT(0 < N);
    NN_STATIC_ASSERT(N <= TouchStateCountMax);
    int32_t count;          //!< 自動操作のタッチの数
    NN_PADDING4;
    TouchState touches[N];  //!< 自動操作のタッチ
};

/**
 * @brief       TouchScreen をファイナライズします。
 *
 * @details     事前に nn::hid::InitializeTouchScreen() を呼び出した回数と同じだけ
 *              呼びだされた際に TouchScreen の初期化状態を解除します。
 *
 * @pre
 *              - TouchScreen が初期化状態にある
 */
void FinalizeTouchScreen() NN_NOEXCEPT;

/**
 * @brief       TouchScreen に自動操作状態を設定します。
 *
 * @tparam      N                           格納するタッチの数
 *
 * @param[in]   value                       TouchScreen の自動操作状態
 *
 * @pre
 *              - TouchScreen が初期化状態にある
 *              - 0 < N && N <= TouchStateCountMax
 */
template<size_t N>
void SetTouchScreenAutoPilotState(const TouchScreenAutoPilotState<N>& value
                                  ) NN_NOEXCEPT;

/**
 * @brief       TouchScreen の自動操作状態を解除します。
 *
 * @pre
 *              - TouchScreen が初期化状態にある
 */
void UnsetTouchScreenAutoPilotState() NN_NOEXCEPT;

}}} // namespace nn::hid::debug
