﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       SixAxisSensor のデバッグ機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_NpadCommonTypes.h>

namespace nn { namespace hid { namespace debug {

/**
 * @brief       加速度センサーのキャリブレーション値のシフト設定を変更します。
 *
 * @details     指定の SixAxisSensorHandle と対応する加速度センサーのキャリブレーション値のシフト設定を変更します。
 *              設定値は本関数の呼び出し以降に接続されたコントローラに対して有効になります。
 *              呼び出し時点で接続されているコントローラは再接続することでシフト設定が反映されます。
 *              シフト設定は任意のアプリ上で有効になり、コントローラの接続切断ではリセットされません。
 *              本体の再起動でシフト設定はクリアされます。
 *
 *              shiftOrigin には、加速度センサーのゼロ点のシフト量を指定します。単位は G です。
 *              shiftSensitivity には、加速度センサーの感度に掛ける係数を指定します。
 *
 *              デフォルトのシフト設定値は以下の通りです。
 *              キャリブレーション値のシフトは行われません。
 *              - shiftOrigin == 0.f
 *              - shiftSensitivity == 1.f
 *
 * @param[in]   handle                  SixAxisSensorHandle です。
 * @param[in]   shiftOrigin             加速度センサーのゼロ点シフト量[G] です。
 * @param[in]   shiftSensitivity        加速度センサーの感度に掛ける係数です。
 */
void SetShiftAccelerometerCalibrationValue(const SixAxisSensorHandle& handle,
                                           float shiftOrigin,
                                           float shiftSensitivity) NN_NOEXCEPT;

/**
 * @brief       加速度センサーのキャリブレーション値のシフト設定を取得します。
 *
 * @details     指定の SixAxisSensorHandle と対応する加速度センサーのキャリブレーション値のシフト設定を取得します。
 *
 * @param[out]   pOutShiftOrigin         加速度センサーのゼロ点シフト量[G] です。
 * @param[out]   pOutShiftSensitivity    加速度センサーの感度に掛ける係数です。
 * @param[in]    handle                  SixAxisSensorHandle です。
 */
void GetShiftAccelerometerCalibrationValue(float* pOutShiftOrigin,
                                           float* pOutShiftSensitivity,
                                           const SixAxisSensorHandle& handle) NN_NOEXCEPT;

/**
 * @brief       ジャイロセンサーのキャリブレーション値のシフト設定を変更します。
 *
 * @details     指定の SixAxisSensorHandle と対応するジャイロセンサーのキャリブレーション値のシフト設定を変更します。
 *              設定値は本関数の呼び出し以降に接続されたコントローラに対して有効になります。
 *              呼び出し時点で接続されているコントローラは再接続することでシフト設定が反映されます。
 *              シフト設定は任意のアプリ上で有効になり、コントローラの接続切断ではリセットされません。
 *              本体の再起動でシフト設定はクリアされます。
 *
 *              shiftOrigin には、ジャイロセンサーのゼロ点のシフト量を指定します。 1.f == 360dps とした単位系で指定します。
 *              shiftSensitivity には、ジャイロセンサーの感度に掛ける係数を指定します。
 *
 *              デフォルトのシフト設定値は以下の通りです。
 *              キャリブレーション値のシフトは行われません。
 *              - shiftOrigin == 0.f
 *              - shiftSensitivity == 1.f
 *
 * @param[in]   handle                  SixAxisSensorHandle です。
 * @param[in]   shiftOrigin             ジャイロセンサーのゼロ点シフト量です。 1.f == 360dps とした単位系で指定します。
 * @param[in]   shiftSensitivity        ジャイロセンサーの感度に掛ける係数です。
 */
void SetShiftGyroscopeCalibrationValue(const SixAxisSensorHandle& handle,
                                       float shiftOrigin,
                                       float shiftSensitivity) NN_NOEXCEPT;

/**
 * @brief       ジャイロセンサーのキャリブレーション値のシフト設定を取得します。
 *
 * @details     指定の SixAxisSensorHandle と対応するジャイロセンサーのキャリブレーション値のシフト設定を取得します。
 *
 * @param[out]   pOutShiftOrigin         ジャイロセンサーのゼロ点シフト量[G] です。
 * @param[out]   pOutShiftSensitivity    ジャイロセンサーの感度に掛ける係数です。
 * @param[in]    handle                  SixAxisSensorHandle です。
 */
void GetShiftGyroscopeCalibrationValue(float* pOutShiftOrigin,
                                       float* pOutShiftSensitivity,
                                       const SixAxisSensorHandle& handle) NN_NOEXCEPT;


}}} // namespace nn::hid::debug
