﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ファームウェア更新に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace hid { namespace debug {

/**
 * @brief       ファームウェア更新中のステージを表す型です。
 */
enum FirmwareUpdateStage
{
    FirmwareUpdateStage_Preparing,                //!< 更新の準備中です。
    FirmwareUpdateStage_BluetoothDownload,        //!< Bluetooth Firmware のダウンロード中です。
    FirmwareUpdateStage_BluetoothVerify,          //!< Bluetooth Firmware  のベリファイ中です。
    FirmwareUpdateStage_BluetoothCommit,          //!< Bluetoooth Firmware を確定しています。
    FirmwareUpdateStage_Reboot1,                  //!< コントローラーの再起動中です。
    FirmwareUpdateStage_McuVersionCheck,          //!< Mcu Firmware のバージョンチェック中です。
    FirmwareUpdateStage_McuBoot,                  //!< Mcu Firmware のシステムブート中です。
    FirmwareUpdateStage_McuErasing,               //!< Mcu Firmware の消去中です。
    FirmwareUpdateStage_McuWriting,               //!< Mcu Firmware の書き込み中です。
    FirmwareUpdateStage_McuReboot,                //!< Mcu Firmware 更新後のリセット中です。
    FirmwareUpdateStage_McuVerify,                //!< Mcu Firmware のベリファイ中です。
    FirmwareUpdateStage_Reboot2,                  //!< コントローラーの再起動中です。
    FirmwareUpdateStage_Completed,                //!< Firmware の更新が完了しました。
};

/**
 * @brief       ファームウェア個別更新の対象となるチップを表す型です。
 */
enum FirmwareUpdateTargetChip : int64_t
{
    FirmwareUpdateTargetChip_Bluetooth,           //!< Bluetooth MCU
    FirmwareUpdateTargetChip_Mcu                  //!< Sub MCU
};

/**
 * @brief       ファームウェアバージョンです
 */
typedef system::FirmwareVersion FirmwareVersion;

//! @name ファームウェア更新 関連 API
//! @{

/**
 * @brief       ファームウェア更新機能を初期化します。
 *
 * @details     ファームウェア更新を開始する前に必ず、本関数を呼び出す必要があります。
 *              本関数が呼ばれると、その時点で接続されている全てのコントローラーが切断されます。
 *
 */
void InitializeFirmwareUpdate() NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新の処理を開始します。
 *
 * @details     UniquePadId で指定されたコントローラーに対してファームウェア更新処理を開始します。
 *              ファームウェア更新中の進捗状況は、GetFirmwareUpdateProgress で取得することができます。
 *              同時に Firmware 更新ができるコントローラーは 1つのみです。
 *
 *              アップデートの要否にかかわらず、システムデータ内のファームウェアへの更新を行います。
 *              必要に応じて、コントローラー MCU ファームウェアの IAP 領域も更新します。
 *
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                              処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,   コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateBusy,      別のコントローラーでファームウェア更新が行われています。}
 *   @handleresult{nn::hid::system::ResultFirmwareImageReadFailed, ファームウェアイメージの読み込みに失敗しました。}
 * @endretresult
 */
::nn::Result StartFirmwareUpdate(system::UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       デバッグ/巻き戻し用データを使用して、ファームウェア更新の処理を開始します。
 *
 * @details     指定したコントローラーについて、ファームウェア更新を開始します。
 *              更新先のファームウェアとしてデバッグ用システムデータ内のものを使用する以外は、 StartFirmwareUpdate() と同様です。
 *
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                              処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,   コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateBusy,      別のコントローラーでファームウェア更新が行われています。}
 *   @handleresult{nn::hid::system::ResultFirmwareImageReadFailed, ファームウェアイメージの読み込みに失敗しました。}
 * @endretresult
 */
::nn::Result StartFirmwareUpdateForRevert(system::UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新の処理を開始します。
 *
 * @details     UniquePadId で指定されたコントローラーに対してファームウェア更新処理を開始します。
 *              ファームウェア更新中の進捗状況は、GetFirmwareUpdateProgress で取得することができます。
 *              同時に Firmware 更新ができるコントローラーは 1つのみです。
 *
 *              書き込むファームウェアのバイナリは、は pImageData と imageSize で指定します。
 *              pImageData が指す領域は、ファームウェアの更新完了まで解放せず保持しておく必要があります。
 *
 * @param[out]  pOutHandle                  ファームウェア更新を開始したコントローラーに対応するハンドル
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 * @param[in]   target                      ファームウェア更新対象のチップ
 * @param[in]   pImageData                  ファームウェアイメージのデータ
 * @param[in]   imageSize                   ファームウェアイメージのバイト数
 *
 * @pre
 *  - target は FirmwareUpdateTargetChip で定義された値
 *  - pImageData != nullptr
 *  - pImageData が nn::os::MemoryPageSize の整数倍
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                              処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,   コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareUpdateBusy,      別のコントローラーでファームウェア更新が行われています。}
 * @endretresult
 */
::nn::Result StartFirmwareUpdateIndividual(system::FirmwareUpdateDeviceHandle* pOutHandle,
                                           system::UniquePadId id,
                                           FirmwareUpdateTargetChip target,
                                           char* pImageData,
                                           size_t imageSize) NN_NOEXCEPT;

/**
 * @brief       ファームウェアバージョンを取得します
 *
 * @details     ファームウェアバージョンを取得します。
 *              ファームウェアバージョンの読み出し処理には時間がかかるため、読み出し中は ResultFirmwareVersionReading が返ります。
 *
 * @param[out]  pOutValue                   読み出されたファームウェアバージョン
 * @param[in]   id                          ファームウェア更新を行う Npad に対する NpadId
 * @param[in]   deviceType                  ファームウェア更新を行うデバイス
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                             処理に成功しました。}
 *   @handleresult{nn::hid::ResultNpadControllerNotConnected,     指定された Npad にコントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::ResultFirmwareVersionReading, ファームウェアバージョンを読み出し中です。}
 * @endretresult
 */
::nn::Result GetFirmwareVersion(FirmwareVersion* pOutValue,
                                NpadIdType id,
                                system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

/**
 * @brief       ファームウェア更新の経過状況を取得します。
 *
 * @details     ファームウェア更新の経過状況を取得します。
 *              ファームウェア更新の経過状況は以下のステージと経過の2種類の値で取得されます。
 *
 *              ステージはファームウェア更新処理の内容を示します。
 *              コントローラーの種類によっても必要なファームウェア更新処理の内容が異なります。
 *
 *              経過は、各ステージにおいてどの程度処理が完了したかと示します。
 *              値は 0 ～ 100 の値として取得されます。
 *
 * @param[out]  pOutStage                   ファームウェア更新中のステージ
 * @param[out]  pOutProgress                ステージ中のファームウェア更新状況
 *
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,        コントローラーが接続されていません。}
 *   @handleresult{nn::hid::system::FirmwareUpdateNotStarted,           ファームウェア更新が開始していません。}
 *   @handleresult{nn::hid::system::FirmwareUpdateBluetoothVerifyError, ファームウェアのベリファイ中にエラーとなりました。}
 *
 */
::nn::Result GetFirmwareUpdateStage(FirmwareUpdateStage* pOutStage,
                                    uint8_t* pOutProgress) NN_NOEXCEPT;

/**
 * @brief       アップデート先のファームウェアバージョンを取得します
 *
 * @details     アップデート先のファームウェアバージョンを取得します。
 *
 * @param[out]  pOutValue                   読み出されたファームウェアバージョン
 * @param[in]   id                          ファームウェア更新を行う Npad に対する NpadId
 * @param[in]   deviceType                  ファームウェア更新を行うデバイス
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                         処理に成功しました。}
 *   @handleresult{nn::hid::ResultNpadControllerNotConnected, 指定された Npad にコントローラーが接続されていません。}
 * @endretresult
 */
::nn::Result GetDestinationFirmwareVersion(FirmwareVersion* pOutValue,
                                           NpadIdType id,
                                           system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

/**
 * @brief       デバッグ/巻き戻し用の更新先ファームウェアのバージョンを取得します。
 *
 * @details     指定したコントローラーに対応する、デバッグ用システムデータ内のファームウェアバージョンを取得します。
 *
 * @param[out]  pOutVersion                 更新先ファームウェアのバージョン
 * @param[in]   id                          ファームウェア更新を行うコントローラーの UniquePadId
 *
 * @pre
 *  - @a pOutVersion != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,    指定した UniquePadId にコントローラーが接続されていません。}
 * @endretresult
 */
::nn::Result GetAvailableFirmwareVersionForRevert(FirmwareVersion* pOutVersion,
                                                  system::UniquePadId id) NN_NOEXCEPT;

/**
 * @brief       コントローラーがファームウェア更新中であるかどうか確認します。
 *
 * @details     指定したコントローラーがファームウェア更新中かどうかを確認します。
 *
 * @param[out]  pOutIsUpdating              ファームウェア更新中かどうか
 * @param[in]   id                          確認するコントローラーの UniquePadId
 *
 * @pre
 *  - @a pOutIsUpdating != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::hid::system::ResultUniquePadDisconnected,    指定した UniquePadId にコントローラーが接続されていません。}
 * @endretresult
 */
::nn::Result IsFirmwareUpdatingDevice(bool* pOutIsUpdating,
                                      system::UniquePadId id) NN_NOEXCEPT;

//! @}

}}} // namespace nn::hid::system
