﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       AbstractedPad のデバッグ機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_BatteryLevel.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/detail/hid_AbstractedPadTypes.h>
#include <nn/hid/system/hid_CommonTypes.h>



namespace nn { namespace hid { namespace debug {

const int AbstractedPadHandleCountMax = 8;  //!< GetAbstractedPadHandles() が一度に返す AbstractedPadHandle の最大数

/**
 * @brief       基本的な構成を持つ AbstractedPad の入力状態の属性定義です。
 */
struct AbstractedPadAttribute
{
    typedef ::nn::util::BitFlagSet<32, AbstractedPadAttribute>::Flag<0>
            IsConnected;    //!< 対応するデバイスがシステムと接続状態にあるか否か
    typedef ::nn::util::BitFlagSet<32, AbstractedPadAttribute>::Flag<1>
            IsSixAxisSensorEnabled;    //!< 対応するデバイスが6軸センサーが有効か否か
};

/**
 * @brief       基本的な構成を持つ AbstractedPad の入力状態の属性集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, AbstractedPadAttribute> AbstractedPadAttributeSet;

/**
 * @brief       基本的な構成を持つ AbstractedPad のハンドルです。
 */
struct AbstractedPadHandle
{
    uint64_t value;
};

/**
 * @brief       基本的な構成を持つ AbstractedVirtualPad の Id です。
 */
enum AbstractedVirtualPadId : int8_t
{
    AbstractedVirtualPadId_No01 = 0x00,
    AbstractedVirtualPadId_No02,
    AbstractedVirtualPadId_No03,
    AbstractedVirtualPadId_No04,
    AbstractedVirtualPadId_No05,
    AbstractedVirtualPadId_No06,
    AbstractedVirtualPadId_No07,
    AbstractedVirtualPadId_No08
};

const int AbstractedVirtualPadIdCountMax = AbstractedVirtualPadId_No08 + 1; //!< AbstractedVirtualPadId の最大数

/**
 * @brief       基本的な構成を持つ AbstractedPad の入力状態を表す構造体です。
 */
struct AbstractedPadState
{
    system::DeviceTypeSet deviceType;           //!< 基本的な構成を持つ AbstractedPad のデバイスタイプです。
    AbstractedPadAttributeSet attributes;       //!< 基本的な構成を持つ AbstractedPad の入力状態の属性です。

    system::ControllerColor color;              //!< 基本的な構成を持つ AbstractedPad のコントローラー色です。
    system::InterfaceType interfaceType;        //!< 基本的な構成を持つ AbstractedPad の接続インターフェースです。
    system::PowerInfo powerInfo;                //!< 基本的な構成を持つ AbstractedPad の電源状態です。

    detail::AbstractedPadButtonSet buttons;     //!< 基本的な構成を持つ AbstractedPad のデジタルボタンの状態です。
    AnalogStickState analogStickL;              //!< 基本的な構成を持つ AbstractedPad の左アナログスティックの状態です。
    AnalogStickState analogStickR;              //!< 基本的な構成を持つ AbstractedPad の右アナログスティックの状態です。

    SixAxisSensorState sixAxisSensorState;
};

/**
 * @brief       仮想デバイスのエミュレート入力を行います。
 *
 * @param[in]   padId          対象デバイスのハンドル
 * @param[in]   value          適用するデバイスの状態
 *
 * @details     仮想的なコントローラーデバイスの状態を更新し、入力のエミュレートを行います。@n
 *              最大 8 台までのデバイスの入力に対応しています。
 */
::nn::Result SetAutoPilotVirtualPadState(const AbstractedVirtualPadId& virtualPadId,
                           const AbstractedPadState& value) NN_NOEXCEPT;

/**
 * @brief       仮想デバイスを切断します。
 *
 * @param[in]   padId          対象デバイスのハンドル
 *
 * @details     仮想コントローラーを切断します。@n
 */

::nn::Result UnsetAutoPilotVirtualPadState(const AbstractedVirtualPadId& virtualPadId) NN_NOEXCEPT;

/**
 * @brief       仮想デバイスを全て切断します。
 *
 * @details     接続されている全ての仮想コントローラーを切断します。
 */
::nn::Result UnsetAllAutoPilotVirtualPadState() NN_NOEXCEPT;

/**
 * @brief       AbstractedPad のハンドルを取得します。
 *
 * @details     接続されている全てのコントローラのハンドルを取得します。
 *              取得できる AbstractedPadHandle の上限は AbstractedPadHandleCountMax 個です。
 *
 * @param[out]  pOutCount                   取得したハンドル数
 * @param[out]  pOutHandles                 AbstractedPadHandle を読みだすバッファ（配列）
 * @param[in]   count                       読みだす AbstractedPad のハンドル数
 *
 */
::nn::Result GetAbstractedPadHandles(int* pOutCount, debug::AbstractedPadHandle* pOutHandles, int count) NN_NOEXCEPT;

/**
 * @brief       AbstractedPad の最新の入力状態を取得します。
 *
 * @param[out]  pOutState                   入力状態
 * @param[in]   handle                      読みだす AbstractedPad のハンドル
 *
 */
::nn::Result GetAbstractedPadState(debug::AbstractedPadState* pOutState, ::nn::hid::debug::AbstractedPadHandle handle) NN_NOEXCEPT;

/**
 * @brief       AbstractedPad の最新の入力状態を取得します。
 *
 * @details     接続されている全てのコントローラのハンドルと最新の入力状態を取得します。
 *              取得できる AbstractedPadHandle の上限は AbstractedPadHandleCountMax 個です。
 *
 * @param[out]  pOutCount                   取得したハンドル数
 * @param[out]  pOutHandles                 AbstractedPadHandle を読みだすバッファ（配列）
 * @param[out]  pOutStates                  入力状態（配列）
 * @param[in]   count                       読みだす AbstractedPad のハンドル数
 *
 */
::nn::Result GetAbstractedPadsState(int* pOutCount, debug::AbstractedPadHandle* pOutHandles, debug::AbstractedPadState* pOutStates, int count) NN_NOEXCEPT;

}}} // namespace nn::hid::debug
