﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <sstream>
#include <vector>

#include <nn/util/util_StringView.h>

#include <nn/gfxTool/gfxTool_Custom.h>
#include <nn/gfxTool/gfxTool_Error.h>
#include <nn/gfxTool/gfxTool_ApiCommon.h>

namespace nn {
namespace gfxTool {

// TODO: VS2013
template< size_t Index, typename R, typename Arg0 >
auto ArgType( R( *)( Arg0 ) ) ->
    typename std::tuple_element< Index, std::tuple< Arg0 > >::type;
template< size_t Index, typename R, typename C, typename Arg0 >
auto ArgType( R( C::* )( Arg0 ) ) ->
    typename std::tuple_element< Index, std::tuple< Arg0 > >::type;

template< size_t Index, typename R, typename Arg0, typename Arg1 >
auto ArgType( R( *)( Arg0, Arg1 ) ) ->
    typename std::tuple_element< Index, std::tuple< Arg0, Arg1 > >::type;
template< size_t Index, typename R, typename C, typename Arg0, typename Arg1 >
auto ArgType( R( C::* )( Arg0, Arg1 ) ) ->
    typename std::tuple_element< Index, std::tuple< Arg0, Arg1 > >::type;

template< typename T >
struct IsBasicString
{
    static const bool value = false;
};
template< typename T, typename U, typename V >
struct IsBasicString< std::basic_string< T, U, V > >
{
    static const bool value = true;
};

template< typename TTo, typename TFrom >
struct ArbitrarilyTypeCaster
{
    static TTo Cast( TFrom from )
    {
        typename Custom< typename std::conditional< std::is_same< TFrom, wchar_t* >::value ||
            std::is_same< TFrom, std::wstring >::value || std::is_same< TTo, std::wstring >::value,
            std::wstringstream, std::stringstream >::type >::Type ss;

        TTo ret;
        if( !( ss << from && ss >> ret && ss.eof() ) )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CastError, "%s -> %s\n",
                typeid( TFrom ).name(), typeid( TTo ).name() );
        }
        return ret;
    }
};

template< typename TTo, typename TFrom >
struct StringCaster
{
    static TTo Cast( TFrom from )
    {
        typename Custom< typename std::conditional< std::is_same< TFrom, wchar_t* >::value ||
            std::is_same< TFrom, std::wstring >::value || std::is_same< TTo, std::wstring >::value,
            std::wstringstream, std::stringstream >::type >::Type ss;

        ss << from;
        return ss.str();
    }
};

template< typename TTo, typename TFrom >
struct AssignCaster
{
    static TTo Cast( TFrom from )
    {
        return from;
    }
};

template< typename TTo, typename TFrom >
inline TTo LexicalCast( const TFrom& from )
{
    // TODO: ちゃんと書く

    typedef typename std::conditional< IsBasicString< TTo >::value,
        StringCaster< TTo, TFrom >, ArbitrarilyTypeCaster< TTo, TFrom > >::type StringArbitarityTypeCaster;
    typedef typename std::conditional< std::is_same< TTo, TFrom >::value,
        AssignCaster< TTo, TFrom >, StringArbitarityTypeCaster >::type Caster;

    return Caster::Cast( from );

}

template< typename TContainer, typename TChar >
inline TContainer TextToArray( const TChar* pText )
{
    typename Custom< std::basic_stringstream< TChar,
        std::char_traits< TChar >, std::allocator< TChar > > >::Type ss;
    ss << pText;
    TContainer ret;
    while( !ss.eof() )
    {
        typename TContainer::value_type value;
        ss >> value;
        if( ss.fail() )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CastError, "%s -> %s[]\n",
                pText, typeid( typename TContainer::value_type ).name() );
        }
        ret.push_back( value );
    }
    return std::move( ret );
}

Custom< std::string >::Type ReadTextFile( const char* pFilename );
bool WriteTextFile( const char* pFilename, const nn::util::string_view& text );

inline bool IsBom( const void* pFileHeader )
{
    auto pBom = static_cast< const uint8_t* >( pFileHeader );
    return pBom[ 0 ] == 0xEF && pBom[ 1 ] == 0xBB && pBom[ 2 ] == 0xBF;
}

Custom< std::vector< char > >::Type Format( const char* pFormat, ... );
Custom< std::vector< char > >::Type Format( const char* pFormat, va_list arg );

template< typename T, typename S, typename Comp >
bool CompareInSrcType( S src, T compVal, Comp comp )
{
    return comp( src, static_cast< S >( compVal ) );
}

template< typename T, typename S, typename Comp, bool Check = true >
struct CastCheckerImpl
{
    static void Check( S src, T compVal, Comp comp )
    {
        if ( !CompareInSrcType( src, compVal, comp ) )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CastError,
                "Cast Error %s( %s ) -> %s",
                typeid( S ).name(), LexicalCast< std::string >( src ).data(), typeid( T ).name() );
        }
    }
};
template< typename T, typename S, typename Comp >
struct CastCheckerImpl< T, S, Comp, false >
{
    static void Check( S, T, Comp )
    {
    }
};

template< typename T, typename S, bool Over, bool Under >
struct CastCheckerOverUnder
{
    static void Check( S src )
    {
        CastCheckerImpl< T, S, std::less_equal< S >, Over >::Check(
            src, std::numeric_limits< T >::max NN_PREVENT_MACRO_FUNC (), std::less_equal< S >() );
        CastCheckerImpl< T, S, std::greater_equal< S >, Under >::Check(
            src, std::numeric_limits< T >::lowest NN_PREVENT_MACRO_FUNC (), std::greater_equal< S >() );
    }
};

template< typename T, typename S >
struct CastChecker
    : public CastCheckerOverUnder< T, S , std::is_floating_point< S >::value || ( sizeof( S ) > sizeof( T ) ) ||
        ( sizeof( S ) == sizeof( T ) && !std::is_signed< S >::value ), std::is_signed< S >::value>
{
};

template< typename T >
struct CastChecker< T, T >
    : public CastCheckerOverUnder< T, T, false, false >
{
};

template< typename T, typename S >
inline T NumericCast( S src )
{
    CastChecker< T, S >::Check( src );
    return static_cast< T >( src );
}

template< typename TTo, typename TFrom >
struct StaticCaster
{
    TTo operator()( TFrom from )
    {
        return static_cast< TTo >( from );
    }
};
template< typename TTo, typename TFrom >
struct NumericCaster
{
    TTo operator()( TFrom from )
    {
        return NumericCast< TTo >( from );
    }
};
template< typename TTo, typename TFrom >
struct LexicalCaster
{
    TTo operator()( TFrom from )
    {
        return LexicalCast< TTo >( from );
    }
};

template< typename TFrom, template< typename, typename > class Caster >
struct AutoCast
{
    explicit AutoCast( TFrom from )
        : value( from )
    {
    }

    TFrom value;

    template< typename T >
    NN_IMPLICIT operator T() const
    {
        return Caster< T, TFrom >()( value );
    }
};

template< typename TFrom >
AutoCast< TFrom, StaticCaster > StaticCastAuto( TFrom from )
{
    return AutoCast< TFrom, StaticCaster >( from );
}
template< typename TFrom >
AutoCast< TFrom, NumericCaster > NumericCastAuto( TFrom from )
{
    return AutoCast< TFrom, NumericCaster >( from );
}
template< typename TFrom >
AutoCast< TFrom, LexicalCaster > LexicalCastAuto( TFrom from )
{
    return AutoCast< TFrom, LexicalCaster >( from );
}

Custom< std::string >::Type GetLastErrorString();

Custom< std::string >::Type GetModulePath( void* hModule );

Custom< std::string >::Type ConvertEncoding(
    const Custom< std::string >::Type& value, int fromCodePage, int toCodePage );

inline void TonngfxToolString( nngfxToolString* pDest, const Custom< std::string >::Type& src )
{
    pDest->pValue = src.c_str();
    pDest->length = NumericCastAuto( src.length() );
}

bool CaseInsensiveEquals( const char* lhs, const char* rhs );

Custom< std::string >::Type AddLineNumber( const nn::util::string_view& source );

}
}
