﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/util/util_Vector.h>

namespace nn {
namespace gfx {
namespace util {


//---------------------------------------------------------------------------
//! @brief 頂点属性です。
//---------------------------------------------------------------------------
enum PrimitiveShapeAttribute
{
    PrimitiveShapeAttribute_Pos = 0,
    PrimitiveShapeAttribute_Normal,
    PrimitiveShapeAttribute_Uv,

    PrimitiveShapeAttribute_CountMax
};

//---------------------------------------------------------------------------
//! @brief 頂点フォーマットです。
//---------------------------------------------------------------------------
enum PrimitiveShapeFormat
{
    PrimitiveShapeFormat_Pos      = 1 << PrimitiveShapeAttribute_Pos,
    PrimitiveShapeFormat_Normal   = 1 << PrimitiveShapeAttribute_Normal,
    PrimitiveShapeFormat_Uv       = 1 << PrimitiveShapeAttribute_Uv,

    PrimitiveShapeFormat_Default = PrimitiveShapeFormat_Pos | PrimitiveShapeFormat_Uv | PrimitiveShapeFormat_Normal
};

//---------------------------------------------------------------------------
//! @brief プリミティブの頂点情報について計算します。
//---------------------------------------------------------------------------
class PrimitiveShape
{
    NN_DISALLOW_COPY(PrimitiveShape);

public:

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~PrimitiveShape();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファのポインタを返します。
    //!
    //! @details このメソッドを実行する前に、Calculateメソッドを実行する必要があります。
    //!
    //! @return ポインタを返します。
    //---------------------------------------------------------------------------
    void* GetIndexBuffer() const;

    //---------------------------------------------------------------------------
    //! @brief  頂点バッファのポインタを返します。
    //!
    //! @details このメソッドを実行する前に、Calculateメソッドを実行する必要があります。
    //!
    //! @return ポインタを返します。
    //---------------------------------------------------------------------------
    void* GetVertexBuffer() const;

    //---------------------------------------------------------------------------
    //! @brief  頂点バッファのストライドのサイズを返します。
    //!
    //! @return ストライドのサイズをバイト単位で返します。
    //---------------------------------------------------------------------------
    size_t GetStride() const;

    //---------------------------------------------------------------------------
    //! @brief  頂点バッファのサイズを返します。
    //!
    //! @return 頂点バッファのサイズをバイト単位で返します。
    //---------------------------------------------------------------------------
    size_t GetVertexBufferSize() const;

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファのサイズを返します。
    //!
    //! @return インデックスバッファのサイズをバイト単位で返します。
    //---------------------------------------------------------------------------
    size_t GetIndexBufferSize() const;

    //---------------------------------------------------------------------------
    //! @brief  頂点フォーマットを返します。
    //!
    //! @return 頂点フォーマットを返します。
    //---------------------------------------------------------------------------
    PrimitiveShapeFormat GetVertexFormat() const;

    //---------------------------------------------------------------------------
    //! @brief  プリミティブトポロジーを返します。
    //!
    //! @return プリミティブトポロジーを返します。
    //---------------------------------------------------------------------------
    nn::gfx::PrimitiveTopology GetPrimitiveTopology() const;

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファのアライメントを返します。
    //!
    //! @return インデックスバッファのアライメントを返します。
    //---------------------------------------------------------------------------
    size_t GetIndexBufferAlignment() const;

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファのフォーマットを返します。
    //!
    //! @return インデックスバッファのフォーマットを返します。
    //---------------------------------------------------------------------------
    IndexFormat GetIndexBufferFormat() const;

    //---------------------------------------------------------------------------
    //! @brief       頂点数を返します。
    //---------------------------------------------------------------------------
    int GetVertexCount() const;

    //---------------------------------------------------------------------------
    //! @brief       インデックス数を返します。
    //---------------------------------------------------------------------------
    int GetIndexCount() const;

    //---------------------------------------------------------------------------
    //! @brief  与えられたバッファに、頂点とインデックスの情報を書き込みます。
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    void Calculate(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

protected:

    //---------------------------------------------------------------------------
    //! @brief       コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTology       プリミティブトポロジーです。
    //---------------------------------------------------------------------------
    PrimitiveShape(PrimitiveShapeFormat       vertexFormat,
                   nn::gfx::PrimitiveTopology primitiveTopology);

    //---------------------------------------------------------------------------
    //! @brief  Calculateメソッドの実装
    //!
    //! @details PrimitiveShapeの継承クラスで実装します。
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize) = 0;

    //---------------------------------------------------------------------------
    //! @brief       頂点バッファのポインタを設定します。
    //!
    //! @param[in]   pVertexBuffer   頂点バッファの先頭アドレス
    //---------------------------------------------------------------------------
    void SetVertexBuffer(void* pVertexBuffer);

    //---------------------------------------------------------------------------
    //! @brief       インデックスバッファのポインタを設定します。
    //!
    //! @param[in]   pIndexBuffer    インデックスバッファの先頭アドレス
    //---------------------------------------------------------------------------
    void SetIndexBuffer(void* pIndexBuffer);

    //---------------------------------------------------------------------------
    //! @brief  頂点バッファのサイズを設定します。
    //!
    //! @param[in]   vertexBufferSize   頂点バッファサイズ
    //---------------------------------------------------------------------------
    void SetVertexBufferSize( size_t vertexBufferSize );

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファのサイズを設定します。
    //!
    //! @param[in]   indexBufferSize   インデックスバッファサイズ
    //---------------------------------------------------------------------------
    void SetIndexBufferSize( size_t indexBufferSize );

    //---------------------------------------------------------------------------
    //! @brief       頂点数を設定します。
    //!
    //! @param[in]   vertexCount   頂点数
    //---------------------------------------------------------------------------
    void  SetVertexCount( int vertexCount );

    //---------------------------------------------------------------------------
    //! @brief       インデックス数を設定します。
    //!
    //! @param[in]   indexCount    インデックス数
    //---------------------------------------------------------------------------
    void SetIndexCount( int indexCount );

private:
    void*                      m_pIndexBuffer;               //!< インデックスバッファの先頭ポインタ
    void*                      m_pVertexBuffer;              //!< 頂点バッファの先頭ポインタ
    PrimitiveShapeFormat       m_VertexFormat;               //!< 頂点バッファのフォーマット
    nn::gfx::IndexFormat       m_IndexBufferFormat;          //!< インデックスバッファのフォーマット
    nn::gfx::PrimitiveTopology m_PrimitiveTopology;          //!< プリミティブトポロジです。
    int                        m_VertexCount;                //!< 頂点数
    int                        m_IndexCount;                 //!< インデックス数
    size_t                     m_VertexBufferSize;           //!< 頂点バッファのサイズ
    size_t                     m_IndexBufferSize;            //!< インデックスバッファのサイズ
};

//---------------------------------------------------------------------------
//! @brief 球の頂点情報について計算します。
//---------------------------------------------------------------------------
class SphereShape : public PrimitiveShape
{
    NN_DISALLOW_COPY(SphereShape);

public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                  　 ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineList
    //! @param[in]   sliceCount            円の分割数です。
    //! @param[in]   stackCount            高さ方向の分割数です。
    //!
    //! @detail 原点を中心として半径 = 1.0 の球を作成します。
    //---------------------------------------------------------------------------
    SphereShape( PrimitiveShapeFormat       vertexFormat,
                 nn::gfx::PrimitiveTopology primitiveTopology,
                 int                        sliceCount,
                 int                        stackCount );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~SphereShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのSphereShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:

    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();

    int m_SliceCount; //! 円の分割数です。
    int m_StackCount; //! 高さ方向の分割数です。
};

//---------------------------------------------------------------------------
//! @brief 円の頂点情報について計算します。
//---------------------------------------------------------------------------
class CircleShape : public PrimitiveShape
{
public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                     ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineStrip
    //! @param[in]   sliceCount            円の分割数です。
    //!
    //! @detail XY 平面上に原点を中心とした半径 1.0 の円を作成します。
    //---------------------------------------------------------------------------
    CircleShape( PrimitiveShapeFormat       vertexFormat,
                 nn::gfx::PrimitiveTopology primitiveTopology,
                 int                        sliceCount );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~CircleShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのCircleShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:
    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();

    int m_SliceCount; //! 円の分割数です。
};

//---------------------------------------------------------------------------
//! @brief キューブの頂点情報について計算します。
//---------------------------------------------------------------------------
class CubeShape : public PrimitiveShape
{
public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                  　 ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineList
    //!
    //! @detail 原点を中心として、横幅 = 1.0, 高さ = 1.0, 奥行き = 1.0 のキューブを作成します。
    //---------------------------------------------------------------------------
    CubeShape( PrimitiveShapeFormat         vertexFormat,
               nn::gfx::PrimitiveTopology   primitiveTopology );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~CubeShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのCubeShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:
    //---------------------------------------------------------------------------
    //! @brief Cubeの頂点数です。
    //---------------------------------------------------------------------------
    enum CubeVertex
    {
        CubeVertexCount_Wired = 8,
        CubeVertexCount_Solid = 24
    };

    //---------------------------------------------------------------------------
    //! @brief Cubeのインデックス数です。
    //---------------------------------------------------------------------------
    enum CubeIndex
    {
        CubeIndexCount_Wired = 48,
        CubeIndexCount_Solid = 36
    };

    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();
};

//---------------------------------------------------------------------------
//! @brief 四角形の頂点情報について計算します。
//---------------------------------------------------------------------------
class QuadShape : public PrimitiveShape
{
public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                  　 ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineStrip
    //!
    //! @detail XY 平面上に原点を中心とした横幅 = 2.0、高さ = 2.0 の四角形を作成します。
    //---------------------------------------------------------------------------
    QuadShape( PrimitiveShapeFormat         vertexFormat,
               nn::gfx::PrimitiveTopology   primitiveTopology );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~QuadShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのQuadShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:
    //---------------------------------------------------------------------------
    //! @brief Quadの頂点数です。
    //---------------------------------------------------------------------------
    enum QuadVertex
    {
        QuadVertexCount = 4
    };

    //---------------------------------------------------------------------------
    //! @brief Quadのインデックス数です。
    //---------------------------------------------------------------------------
    enum QuadIndex
    {
        QuadIndexCount_Wired = 5,
        QuadIndexCountt_Solid = 6
    };

    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();
};

//---------------------------------------------------------------------------
//! @brief 半球の頂点情報について計算します。
//---------------------------------------------------------------------------
class HemiSphereShape : public PrimitiveShape
{
public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                  　 ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineList
    //! @param[in]   sliceCount            半球の分割数です。
    //! @detail  原点を中心とし、Y 軸正の方向を向いた半径 = 1.0fの半球を作成します。
    //---------------------------------------------------------------------------
    HemiSphereShape( PrimitiveShapeFormat         vertexFormat,
                     nn::gfx::PrimitiveTopology   primitiveTopology,
                     int                          sliceCount );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~HemiSphereShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのHemiSphereShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:
    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();

    int m_SliceCount;     //! 半球の分割数です。
};


//---------------------------------------------------------------------------
//! @brief パイプの頂点情報について計算します。
//---------------------------------------------------------------------------
class PipeShape : public PrimitiveShape
{
public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                  　 ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineList
    //! @param[in]   sliceCount            パイプの分割数です。
    //!
    //! @detail 底面が原点を中心とした XZ 平面上に存在し、半径 = 1.0, 高さ = 1.0 のパイプを作成します。
    //---------------------------------------------------------------------------
    PipeShape( PrimitiveShapeFormat           vertexFormat,
                   nn::gfx::PrimitiveTopology primitiveTopology,
                   int                        sliceCount );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~PipeShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのPipeShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:
    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();

    int m_SliceCount;     //! 円の分割数です。
};

//---------------------------------------------------------------------------
//! @brief シリンダーの頂点情報について計算します。
//---------------------------------------------------------------------------
class CylinderShape : public PrimitiveShape
{
public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                  　 ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineList
    //! @param[in]   sliceCount            シリンダーの分割数です。
    //!
    //! @detail 底面が原点を中心とした XZ 平面上に存在し、半径 = 1.0, 高さ = 1.0 のシリンダーを作成します。
    //---------------------------------------------------------------------------
    CylinderShape( PrimitiveShapeFormat           vertexFormat,
                   nn::gfx::PrimitiveTopology     primitiveTopology,
                   int                            sliceCount );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~CylinderShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのCylinderShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:
    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();

    int m_SliceCount;     //! 円の分割数です。
};

//---------------------------------------------------------------------------
//! @brief コーンの頂点情報について計算します。
//---------------------------------------------------------------------------
class ConeShape : public PrimitiveShape
{
public:

    //---------------------------------------------------------------------------
    //! @brief コンストラクタです。
    //!
    //! @param[in]   vertexFormat          頂点バッファのフォーマット
    //! @param[in]   primitiveTopology     プリミティブトポロジーです。
    //!                                  　 ポリゴン: PrimitiveTopology_TriangleList
    //!                                   　ワイヤー: PrimitiveTopology_LineList
    //! @param[in]   sliceCount            円の分割数です。
    //!
    //! @detail 底面が原点を中心とした XZ 平面上に存在し、半径 = 1.0, 高さ = 1.0 のコーンを作成します。
    //---------------------------------------------------------------------------
    ConeShape( PrimitiveShapeFormat         vertexFormat,
               nn::gfx::PrimitiveTopology   primitiveTopology,
               int                          sliceCount );

    //---------------------------------------------------------------------------
    //! @brief  デストラクタ
    //---------------------------------------------------------------------------
    virtual ~ConeShape();

protected:

    //---------------------------------------------------------------------------
    //! @brief  頂点数を計算します。
    //!
    //! @return 頂点数を返します。
    //---------------------------------------------------------------------------
    int CalculateVertexCount();

    //---------------------------------------------------------------------------
    //! @brief  インデックス数を計算します。
    //!
    //! @return インデックス数を返します。
    //---------------------------------------------------------------------------
    int CalculateIndexCount();

    //---------------------------------------------------------------------------
    //! @brief  CalculateメソッドのConeShapeクラスでの実装
    //!
    //! @param[out]  pVertexMemory  頂点情報を書き込むメモリのポインタ
    //! @param[in]   vertexSize     確保済みのメモリサイズ(バイト単位)
    //! @param[out]  pIndexMemory   インデックス情報を書き込むメモリのポインタ
    //! @param[in]   indexSize      確保済みのメモリサイズ(バイト単位)
    //---------------------------------------------------------------------------
    virtual void CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize);

private:

    //---------------------------------------------------------------------------
    //! @brief   頂点バッファを計算します。
    //!
    //! @details SetVertexBufferメソッドで、バッファをセットしておく必要があります。
    //!
    //! @return  頂点バッファの末尾を返します。
    //---------------------------------------------------------------------------
    void* CalculateVertexBuffer();

    //---------------------------------------------------------------------------
    //! @brief  インデックスバッファを計算します。
    //!
    //! @details SetIndexBufferメソッドで、バッファをセットしておく必要があります。
    //---------------------------------------------------------------------------
    template <typename T>
    void CalculateIndexBuffer();

    int m_SliceCount;     //! 円の分割数です。
};

} // namespace util
} // namespace gfx
} // namespace nn
