﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Macro.h>

#include <nn/gfx/gfx_Common.h>
#include <nn/gfx/gfx_Variation-api.gx.h>
#include <nn/gfx/gfx_Enum.h>

#include <nn/gfx/detail/gfx_Declare.h>

struct _GX2Sampler;
struct _GX2Surface;
struct _GX2Texture;
struct _GX2ColorBuffer;
struct _GX2DepthBuffer;

namespace nn {
namespace gfx {

/**
* @brief GX2 API との相互運用を行うためのクラスです。
*
* @details
* 元のオブジェクトの作成者がオブジェクトの所有権をもちます。@n
* Initialize が呼ばれていない gfx オブジェクトに対して Finalize を呼ぶことはできません。
*/
template<>
class TInteroperation< ApiVariationGx2 >
{
    NN_DISALLOW_COPY( TInteroperation );
    TInteroperation() NN_NOEXCEPT;

public:
    /**
    * @brief 低レベルグラフィックス API のターゲットです。
    */
    typedef ApiVariationGx2 Target;

    /**
    * @brief 指定された GX2 のバッファを gfx のバッファに変換します。
    *
    * @param[out] pOutGfxBuffer 変換された gfx のバッファを受け取るためのポインタ
    * @param[in] pGx2Buffer 変換する GX2 のバッファへのポインタ
    * @param[in] bufferSize バイトでのバッファのサイズ
    *
    * @pre
    * - pOutGfxBuffer != NULL
    * - pOutGfxBuffer が初期化されていない
    * - pGx2Buffer != NULL
    */
    static void ConvertToGfxBuffer( TBuffer< Target >* pOutGfxBuffer,
        void* pGx2Buffer, size_t bufferSize ) NN_NOEXCEPT;

    /**
    * @brief 指定された GX2 のサンプラを gfx のサンプラに変換します。
    *
    * @param[out] pOutGfxSampler 変換された gfx のサンプラを受け取るためのポインタ
    * @param[in] pGx2Sampler 変換する GX2 のサンプラへのポインタ
    *
    * @pre
    * - pOutGfxSampler != NULL
    * - pOutGfxSampler が初期化されていない
    * - pGx2Sampler != NULL
    */
    static void ConvertToGfxSampler( TSampler< Target >* pOutGfxSampler,
        _GX2Sampler* pGx2Sampler ) NN_NOEXCEPT;

    /**
    * @brief 指定された GX2 のサーフェイスを gfx のテクスチャに変換します。
    *
    * @param[out] pOutGfxTexture 変換された gfx のテクスチャを受け取るためのポインタ
    * @param[in] pGx2Surface 変換する GX2 のサンプラへのポインタ
    *
    * @pre
    * - pOutGfxTexture != NULL
    * - pGx2Surface != NULL
    */
    static void ConvertToGfxTexture( TTexture< Target >* pOutGfxTexture,
        _GX2Surface* pGx2Surface ) NN_NOEXCEPT;

    /**
    * @brief 指定された GX2 のテクスチャを gfx のテクスチャビューに変換します。
    *
    * @param[out] pOutGfxTextureView 変換された gfx のテクスチャビューを受け取るためのポインタ
    * @param[in] pGx2Texture 変換する GX2 のテクスチャへのポインタ
    *
    * @pre
    * - pOutGfxTextureView != NULL
    * - pOutGfxTextureView が初期化されていない
    * - pGx2Texture != NULL
    */
    static void ConvertToGfxTextureView( TTextureView< Target >* pOutGfxTextureView,
        _GX2Texture* pGx2Texture ) NN_NOEXCEPT;

    /**
    * @brief 指定された GX2 のカラーバッファを gfx のカラーターゲットビューに変換します。
    *
    * @param[out] pOutGfxColorTargetView 変換された gfx のカラーターゲットビューを受け取るためのポインタ
    * @param[in] pGx2ColorBuffer 変換する GX2 のカラーバッファ
    *
    * @pre
    * - pOutGfxColorTargetView != NULL
    * - pOutGfxColorTargetView が初期化されていない
    * - pGx2Texture != NULL
    */
    static void ConvertToGfxColorTargetView( TColorTargetView< Target >* pOutGfxColorTargetView,
        _GX2ColorBuffer* pGx2ColorBuffer ) NN_NOEXCEPT;

    /**
    * @brief 指定された GX2 の深度バッファを gfx の深度ステンシルビューに変換します。
    *
    * @param[out] pOutGfxDepthStencilView 変換された gfx の深度ステンシルビューを受け取るためのポインタ
    * @param[in] pGx2DepthBuffer 変換する GX2 の深度バッファ
    *
    * @pre
    * - pOutGfxDepthStencilView != NULL
    * - pOutGfxDepthStencilView が初期化されていない
    * - pGx2DepthBuffer != NULL
    */
    static void ConvertToGfxDepthStencilView( TDepthStencilView< Target >* pOutGfxDepthStencilView,
        _GX2DepthBuffer* pGx2DepthBuffer ) NN_NOEXCEPT;
};

}
}
