﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace ens {

/**
 * @brief   シンプルな MessagePack リーダー
 *
 * @details
 *  表現できるデータ構造を少なくする代わりに、インターフェイスがシンプルな MessagePack のリーダーです。@n
 *  値の取得毎にデータ全体を走査します。
 *
 *  値を取得する API に指定するパスは以下のように記述します。
 *
 *  - ルート
 *      - "$" を指定
 *  - キー
 *      - ルートの後に "." とキー名を連結して指定
 *      - 例： "$.key1.key2"
 *  - 配列
 *      - "[num]" を指定
 *      - 例： "$.key1.key2.key3[1]"
 *
 *  パスは UTF-8 でエンコードされています。
 *
 *  以下にパスの記述例を示します。
 *
 *  記述例１：文字列単体の MessagePack
 *
 *  - "$"
 *
 *  記述例２：数値の配列の MessagePack
 *
 *  - "$[0]"
 *  - "$[1]"
 *  - "$[2]"
 *
 *  記述例３：構造体 "struct {int a; int b;}" の MessagePack
 *
 *  - "$.a"
 *  - "$.b"
 *
 *  記述例４：配列を含む複雑な構造体 "struct {struct {int x; int y;} points[3]; int color;}" の MessagePack
 *
 *  - "$.points[0].x"
 *  - "$.points[0].y"
 *  - "$.points[1].x"
 *  - "$.points[1].y"
 *  - "$.points[2].x"
 *  - "$.points[2].y"
 *  - "$.color"
 */
class SimpleMessagePackReader
{
private:
    NN_DISALLOW_COPY(SimpleMessagePackReader);
    NN_DISALLOW_MOVE(SimpleMessagePackReader);

public:
    /**
     * @brief   コンストラクタ
     *
     * @param[in]   pData   データ
     * @param[in]   size    データサイズ
     *
     * @pre
     *  - pData != nullptr
     *  - size > 0
     *
     * @details
     */
    SimpleMessagePackReader(const void* pData, size_t size) NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（論理値）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(bool* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(int8_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(int16_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(int32_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(int64_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(uint8_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(uint16_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(uint32_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（整数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(uint64_t* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（浮動小数点数）
     *
     * @param[out]  pOutValue   値
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - pPath != nullptr
     *
     * @details
     */
    bool Get(double* pOutValue, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（文字列）
     *
     * @param[out]  pOutValue   値
     * @param[in]   size        pOutValue のサイズ
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutValue != nullptr
     *  - size > 0
     *  - pPath != nullptr
     *
     * @details
     *  本関数は、サイズが不足した場合、値を格納せずに false を返します。
     *
     *  取得した文字列値は NULL 終端されます。
     */
    bool Get(char* pOutValue, size_t size, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   値を取得します。（バイナリ）
     *
     * @param[out]  pOutSize    取得したサイズ
     * @param[out]  pOutValue   値
     * @param[in]   size        pOutValue のサイズ
     * @param[in]   pPath       パス
     *
     * @return  値を取得できたかどうか
     *
     * @pre
     *  - pOutSize != nullptr
     *  - pOutValue != nullptr
     *  - size > 0
     *  - pPath != nullptr
     *
     * @details
     *  本関数は、サイズが不足した場合、値を格納せずに false を返します。
     */
    bool Get(size_t* pOutSize, void* pOutValue, size_t size, const char* pPath) const NN_NOEXCEPT;

    /**
     * @brief   妥当な MessagePack かどうかを判定します。
     *
     * @return  妥当な MessagePack かどうか
     *
     * @details
     */
    bool IsValid() const NN_NOEXCEPT;

private:
    //
    const nn::Bit8* m_pData;
    //
    size_t m_Size;
};

}}
