﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/ens_Types.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_ThreadTypes.h>

namespace nn { namespace ens {

/**
 * @brief   サービススレッドの実行コンテキストを管理するクラス
 *
 * @details
 *  サービススレッドの実行コンテキストの管理を行うためのクラスです。
 *
 *  サービススレッドを複数並列で実行する場合、1 スレッドにつき 1 つの ServiceThreadContext クラスインスタンスが必要です。
 */
class ServiceThreadContext
{
private:
    NN_DISALLOW_COPY(ServiceThreadContext);
    NN_DISALLOW_MOVE(ServiceThreadContext);

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    ServiceThreadContext() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    ~ServiceThreadContext() NN_NOEXCEPT;

    /**
     * @brief   バッファを設定します。
     *
     * @param[in]   pBuffer     バッファ
     * @param[in]   bufferSize  pBuffer のサイズ
     *
     * @pre
     *  - pBuffer != nullptr
     *  - bufferSize > 0
     *
     * @details
     *  リクエストの実行に必要なバッファを設定します。
     *
     *  一度のリクエストで扱う通信データサイズ（送信サイズまたは受信サイズのより大きい方）以上のバッファを指定します。
     */
    void SetBuffer(void* pBuffer, size_t bufferSize) NN_NOEXCEPT;

public:
    // internal use only
    nn::os::Event& GetStopEvent() NN_NOEXCEPT;
    // internal use only
    nn::os::Event& GetStartedEvent() NN_NOEXCEPT;
    // internal use only
    nn::os::Event& GetStoppedEvent() NN_NOEXCEPT;

    // internal use only
    void SetThread(nn::os::ThreadType* pThread) NN_NOEXCEPT;
    // internal use only
    nn::os::ThreadType* GetThread() const NN_NOEXCEPT;

    // internal use only
    void* GetBuffer() NN_NOEXCEPT;
    // internal use only
    size_t GetBufferSize() const NN_NOEXCEPT;

private:
    //
    nn::os::Event m_StopEvent;
    nn::os::Event m_StartedEvent;
    nn::os::Event m_StoppedEvent;
    //
    nn::os::ThreadType* m_pThread;
    //
    void* m_pBuffer;
    size_t m_BufferSize;
};

}}
