﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/util/rapidjson/ens_RapidJson.h>

namespace nn { namespace ens { namespace detail { namespace util { namespace rapidjson {

/**
 * @brief   RapidJSON 用の入力ストリーム
 *
 * @details
 *  本クラスは RapidJSON の入力ストリームに必要なメソッドを実装しています。@n
 *  本クラスの利用者は、 FillImpl() を実装する必要があります。
 *
 *  本クラスは作業用バッファを持たないため、利用時に SetBuffer() でバッファを指定する必要があります。
 */
class RapidJsonInputStream
{
public:
    // UTF-8 をサポートするため、 char とする。
    typedef char Ch;

public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    RapidJsonInputStream() NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    virtual ~RapidJsonInputStream() NN_NOEXCEPT;

    /**
     * @brief   バッファを設定します。
     *
     * @param[in]   pStringBuffer       文字列バッファ
     * @param[in]   stringBufferSize    文字列バッファのサイズ
     * @param[in]   pStreamBuffer       入力ストリームバッファ
     * @param[in]   streamBufferSize    入力ストリームバッファのサイズ
     *
     * @pre
     *  - pStringBuffer != nullptr
     *  - stringBufferSize > 0
     *  - pStreamBuffer != nullptr
     *  - streamBufferSize > 0
     *
     * @details
     *  文字列バッファのサイズは読み込む JSON に含まれる文字列データの中で一番大きいサイズのものを指定してください。@n
     *  文字列バッファに収まらない文字列値が出現した場合、読み込み処理は失敗します。
     *
     *  入力ストリームバッファは入力元から受け取ったデータを一時的に格納するバッファです。@n
     *  サイズが大きい程一度に受け取るサイズが増えるため、サイズを大きくするとパフォーマンスが向上する場合があります。
     */
    void SetBuffer(Ch* pStringBuffer, size_t stringBufferSize, Ch* pStreamBuffer, size_t streamBufferSize) NN_NOEXCEPT;

    /**
     * @brief   最後に発生したエラーを取得します。
     *
     * @return  最後に発生したエラー
     *
     * @details
     */
    nn::Result GetLastError() const NN_NOEXCEPT;

    /**
     * @brief   文字列のオーバーフローを検知したかどうかを取得します。
     *
     * @return  文字列のオーバーフローを検知したかどうか
     *
     * @details
     *  JSON データ内に SetBuffer() で指定した文字列バッファに収まらない文字列が存在した場合、 true を返します。
     */
    bool IsStringOverflowDetected() const NN_NOEXCEPT;

public:
    /**
     * @brief   読み込み位置を移動せずに入力ストリームの先頭文字を取得します。
     *
     * @return  先頭文字
     *
     * @details
     *  本関数は RapidJSON 用関数です。
     */
    Ch Peek() NN_NOEXCEPT;

    /**
     * @brief   入力ストリームの先頭文字を取得します。
     *
     * @return  先頭文字
     *
     * @details
     *  本関数は RapidJSON 用関数です。
     */
    Ch Take() NN_NOEXCEPT;

    /**
     * @brief   読み込み位置を取得します。
     *
     * @return  読み込み位置
     *
     * @details
     *  本関数は RapidJSON 用関数です。
     */
    size_t Tell() const NN_NOEXCEPT;

    /**
     * @brief   文字列値の入力を開始します。
     *
     * @return  文字列バッファの先頭
     *
     * @details
     *  本関数は RapidJSON 用関数です。@n
     *  RAPIDJSON_NAMESPACE::kParseInsituFlag を指定したパースでは、本関数を使用して文字列バッファの初期化を行います。
     */
    Ch* PutBegin() NN_NOEXCEPT;

    /**
     * @brief   文字列値を入力します。
     *
     * @param[in]   c   文字
     *
     * @details
     *  本関数は RapidJSON 用関数です。@n
     *  RAPIDJSON_NAMESPACE::kParseInsituFlag を指定したパースでは、本関数を使用して文字列を入力します。
     */
    void Put(Ch c) NN_NOEXCEPT;

    /**
     * @brief   文字列値の入力を終了します。
     *
     * @param[in]   pBegin  文字列バッファの先頭
     *
     * @return  入力文字数
     *
     * @pre
     *  - pBegin != nullptr
     *
     * @details
     *  本関数は RapidJSON 用関数です。
     */
    size_t PutEnd(Ch* pBegin) NN_NOEXCEPT;

protected:
    /**
     * @brief   入力ストリームにデータを書き込みます。
     *
     * @param[out]  pOutRead    読み込んだバイト数。
     * @param[out]  pOutData    入力ストリームのデータ。
     * @param[in]   dataSize    入力ストリームのデータサイズ。
     *
     * @return  処理結果
     *
     * @pre
     *  - pOutRead != nullptr
     *  - pOutData != nullptr
     *  - dataSize > 0
     *
     * @details
     */
    virtual nn::Result FillImpl(size_t* pOutRead, void* pOutData, size_t dataSize) NN_NOEXCEPT = 0;

private:
    //
    Ch* m_pString;
    size_t m_StringBufferSize;
    //
    Ch* m_pStream;
    size_t m_StreamBufferSize;
    //
    size_t m_StringSize;
    //
    size_t m_Read;
    size_t m_ReadTotal;
    size_t m_PositionInStream;
    //
    nn::Result m_LastError;
    //
    bool m_IsStringOverflowDetected;

private:
    //
    bool Fill() NN_NOEXCEPT;
};

}}}}}
