﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/detail/util/ens_ResponseReader.h>

namespace nn { namespace ens { namespace detail { namespace util {

/**
 * @brief   レスポンスからデータ構造を読み込むクラス
 *
 * @details
 */
class ResponseStructureReaderImpl
{
public:
    // internal use only
    struct Structure
    {
        const char* pJsonPath;
        ValueType valueType;
        size_t* pOutSize;
        void* pOutValue;
        size_t size;
        int8_t radix;
        bool isFound;
    };

public:
    /**
     * @brief   コンストラクタ
     *
     * @param[in]   pList   構造リスト
     * @param[in]   num     構造リストの要素数
     *
     * @pre
     *  - pList != nullptr
     *  - num > 0
     *
     * @details
     */
    ResponseStructureReaderImpl(Structure* pList, size_t num) NN_NOEXCEPT;

    /**
     * @brief   bool 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, bool* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   int8_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, int8_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   int16_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, int16_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   int32_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, int32_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   int64_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, int64_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   uint8_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, uint8_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   uint16_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, uint16_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   uint32_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, uint32_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   uint64_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, uint64_t* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   uint64_t 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     * @param[in]   radix       基数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *  - 2 <= radix <= 36
     *
     * @details
     *  本関数で変数を追加した場合、文字列を数値に変換します。
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, uint64_t* pOutValue, int8_t radix) NN_NOEXCEPT;

    /**
     * @brief   double 型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, double* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   文字列型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     * @param[in]   size        文字列バッファのサイズ
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *  - size > 0
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, char* pOutValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   バイナリ型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutSize    取得したサイズ
     * @param[out]  pOutValue   出力変数
     * @param[in]   size        バイナリバッファのサイズ
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutSize != nullptr
     *  - pOutValue != nullptr
     *  - size > 0
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, size_t* pOutSize, void* pOutValue, size_t size) NN_NOEXCEPT;

    /**
     * @brief   バイナリ型の変数を追加します。
     *
     * @param[in]   pJsonPath   JSON パス文字列
     * @param[out]  pOutValue   出力変数
     *
     * @pre
     *  - pJsonPath != nullptr
     *  - pOutValue != nullptr
     *
     * @details
     *
     * @attention
     *  JSON パス文字列は、追加済みの変数の JSON パス文字列と重複してはいけません。
     */
    void Add(const char* pJsonPath, ReceiveBuffer* pOutValue) NN_NOEXCEPT;

    /**
     * @brief   レスポンスからデータ構造を読み込みます。
     *
     * @tparam      Reader  ResponseReader を継承したクラス
     *
     * @param[in]   pData   レスポンス
     * @param[in]   size    レスポンスのサイズ
     *
     * @return  妥当なレスポンスを読み込んだかどうか
     *
     * @pre
     *  - pData != nullptr
     *
     * @details
     */
    template <class Reader>
    bool Read(const void* pData, size_t size) NN_NOEXCEPT
    {
        Reader reader;

        reader.SetCallback(ResponseReadCallback, this);

        NN_DETAIL_ENS_RETURN_IF_FALSE(reader.Read(pData, size));

        for (int i = 0; i < m_Count; i++)
        {
            if (!m_pList[i].isFound)
            {
                return false;
            }
        }

        return true;
    }

private:
    //
    Structure* m_pList;
    const size_t m_Num;
    //
    int m_Count;

private:
    //
    void AddImpl(ValueType valueType, const char* pJsonPath,
        size_t* pOutSize, void* pOutValue, size_t size, int8_t radix = 0) NN_NOEXCEPT;

private:
    //
    static bool ResponseReadCallback(const JsonPath& jsonPath, const DataHolder& holder, void* pParam) NN_NOEXCEPT;
};

/**
 * @brief   レスポンスからデータ構造を読み込むクラス
 *
 * @tparam      N   要素数
 *
 * @details
 */
template <size_t N>
class ResponseStructureReader : public ResponseStructureReaderImpl
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     */
    ResponseStructureReader() NN_NOEXCEPT
        : ResponseStructureReaderImpl(m_List, N)
    {
    }

private:
    //
    Structure m_List[N];
};

}}}}
