﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <type_traits>

namespace nn { namespace ens { namespace detail {

/**
 * @brief   ランタイム型情報
 *
 * @details
 */
struct RuntimeTypeInfo
{
    const RuntimeTypeInfo* pParent;

    /**
     * @brief   dynamic_cast 可能かどうかを判定します。
     *
     * @param[in]   pTo 変換先のランタイム型情報
     *
     * @return  dynamic_cast 可能かどうか
     *
     * @pre
     *  - pTo != nullptr
     *
     * @details
     */
    bool CanDynamicCast(const RuntimeTypeInfo* pTo) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTo);

        const RuntimeTypeInfo* pFrom = this;

        while (pFrom)
        {
            if (pFrom == pTo)
            {
                return true;
            }

            pFrom = pFrom->pParent;
        }

        return false;
    }
};

/**
 * @brief   dynamic_cast を行います。
 *
 * @tparam      ToPtr   変換先の型のポインタ
 * @tparam      From    変換元の型
 *
 * @param[in]   pFrom   変換元
 *
 * @return  変換できたら変換先のポインタを、変換できない場合 nullptr を返します。
 *
 * @details
 *  本関数は NN_DETAIL_ENS_DEFINE_RUNTIME_TYPEINFO(_ROOT) でランタイム型情報を定義した場合のみ使用可能です。
 *
 *  変換元のクラスが直接ランタイム型情報を定義していなくても、その継承元がランタイム型情報を定義している場合、 dynamic_cast 可能です。@n
 *  ただし、変換先に指定したクラスがランタイム型情報を直接定義していない場合、コンパイルエラーになります。
 */
template<typename ToPtr, typename From>
ToPtr DynamicCast(From* pFrom) NN_NOEXCEPT
{
    using To = typename std::remove_cv<typename std::remove_pointer<ToPtr>::type>::type;

    // 変換先がランタイム型情報を定義していることを確認する。
    NN_STATIC_ASSERT((std::is_same<To, typename To::SelfType>::value));

    if (pFrom && pFrom->GetRuntimeTypeInfo()->CanDynamicCast(To::GetRuntimeTypeInfoStatic()))
    {
        return static_cast<ToPtr>(pFrom);
    }

    return nullptr;
}

}}}

/**
 * @brief   ランタイム型情報を定義します。（ルートクラス用）
 *
 * @param[in]   Self    継承先のクラス型
 *
 * @details
 */
#define NN_DETAIL_ENS_DEFINE_RUNTIME_TYPEINFO_ROOT(Self) \
    using SelfType = Self;                                                                          \
                                                                                                    \
    static const nn::ens::detail::RuntimeTypeInfo* GetRuntimeTypeInfoStatic() NN_NOEXCEPT           \
    {                                                                                               \
        NN_FUNCTION_LOCAL_STATIC(const nn::ens::detail::RuntimeTypeInfo, s_TypeInfo = {nullptr});   \
        return &s_TypeInfo;                                                                         \
    }                                                                                               \
                                                                                                    \
    virtual const nn::ens::detail::RuntimeTypeInfo* GetRuntimeTypeInfo() const NN_NOEXCEPT          \
    {                                                                                               \
        return GetRuntimeTypeInfoStatic();                                                          \
    }

/**
 * @brief   ランタイム型情報を定義します。
 *
 * @param[in]   Self    継承先のクラス型
 * @param[in]   Base    継承元のクラス型
 *
 * @details
 */
#define NN_DETAIL_ENS_DEFINE_RUNTIME_TYPEINFO(Self, Base) \
    using SelfType = Self;                                                                                                  \
                                                                                                                            \
    static const nn::ens::detail::RuntimeTypeInfo* GetRuntimeTypeInfoStatic() NN_NOEXCEPT                                   \
    {                                                                                                                       \
        NN_FUNCTION_LOCAL_STATIC(const nn::ens::detail::RuntimeTypeInfo, s_TypeInfo = {Base::GetRuntimeTypeInfoStatic()});  \
        return &s_TypeInfo;                                                                                                 \
    }                                                                                                                       \
                                                                                                                            \
    virtual const nn::ens::detail::RuntimeTypeInfo* GetRuntimeTypeInfo() const NN_NOEXCEPT NN_OVERRIDE                      \
    {                                                                                                                       \
        return GetRuntimeTypeInfoStatic();                                                                                  \
    }
